/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file debug.h
 *\brief All debug related features to be used with the High Level API are defined here.
 */

#ifndef SENSOR_DEBUG
#define SENSOR_DEBUG
#ifndef __KERNEL__
#include <stdio.h>
#include "host_serial.h"
#endif
#include "definitions.h"
//Debug Enable Switch
#define DEBUG_ENABLE                             1  /*!< Set to 1 to enable Function Logging feature */
#define I2CLOG_ENABLE                            0  /*!< Set to 1 to enable I2C Logging feature */

/* Error code definition */
#define SENSOR_ERROR_NONE                        0  /*!< No error, Function executed successfully */
#define SENSOR_ERROR                             1  /*!< Generic error code */

#define COMMON_ERROR_BASE                       0x0100 /*!< Base number for all errors common to all function */
#define COMMON_INVALID_PARAMS                   COMMON_ERROR_BASE +0x001 /*!< Provided Invalid params */
#define COMMON_INVALID_OUTPUT                   COMMON_ERROR_BASE +0x002 /*!< Provided Invalid output */
#define SYSTEM_ERROR_BASE                       0x0200 /*!< Base number for all errors related to system function */

#define RANGE_ERROR_BASE                        0x0300 /*!< Base number for all errors related to ranging function */

#define ALS_ERROR_BASE                          0x0400 /*!< Base number for all errors related to als function */

#define DEBUG_SESSION_NOT_OPENE                 2
#define DEBUG_ONLINE                            0x10 /*!< Run time debugging. output via usb-serial port */
#define DEBUG_OFFLINE                           0x20 /*!< Offline debugging. Store debug log in memory space */

/* Debugging api definition */
#define MINIMUM_LOG_LEVEL                       0
#define NORMAL_LOG_LEVEL                        1
#define VERBOSE_LOG_LEVEL                       2
#define I2C_LOG_LEVEL                           3


#ifdef DEBUG_ENABLE

extern char*    pDebugBuffer;
extern uint32_t DebugLogSize;
extern bool_t   DebugStarted;

/* Defines for external use */
#define SPRINTF(buf,fmt,args...) sprintf((char *)buf,fmt,##args)

#define LOG_FUNCTION_START(...)  void* _pLogAllARgs_[]={ __VA_ARGS__ }; loggingFunctionStart(__func__, _pLogAllARgs_);
#define LOG_FUNCTION_END(returned)   loggingFunctionEnd(__func__, returned, _pLogAllARgs_)


#define DEBUG_WRITE_IN_LOG(...) SPRINTF(pDebugBuffer,##__VA_ARGS__); loggingOutput(pDebugBuffer);
#define INTERNAL_DEBUG_LOG(...) if(DebugStarted){logDebugMessageStart(__func__);DEBUG_WRITE_IN_LOG(__VA_ARGS__);logDebugMessageEnd();}
#define INTERNAL_ERROR_LOG(...) logErrorMessageStart(__func__);DEBUG_WRITE_IN_LOG(__VA_ARGS__);logErrorMessageEnd();

#define DEBUG_LOG(...)   INTERNAL_DEBUG_LOG(__VA_ARGS__);CUSTOMER_DEBUG_LOG(__VA_ARGS__)
#define SENSOR_ERROR_LOG(...)  INTERNAL_ERROR_LOG(__VA_ARGS__);CUSTOMER_ERROR_LOG(__VA_ARGS__)

/*  Help compiling in C++  */
#ifdef __cplusplus
extern "C"{
#endif   /*__cplusplus*/


/*!
 * \fn          sensor_error loggingOpen()
 * \brief       Initialize debug sequence
 *
 * This function should be called at the beginning of the debugging session. It initializes everything that is necessary.
 * This function allocates a big size buffer in order to store all logged data.
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingOpen();

/*!
 * \fn          sensor_error loggingClose()
 * \brief       Initialize debug sequence
 *
 * This function shall be called once logging functionalities are no more required.
 * This function will free the memory allocated during the call to logingOpen function
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingClose();

/*!
 * \fn          sensor_error loggingSetMode()
 * \brief       Set up debug mode
 *
 * This function shall be called before start logging functions.
 * This function will set the debug logging mode as realtime output or offline mode.
 * \param[in]   mode
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingSetMode(char mode);

/*!
 * \fn          sensor_error loggingSetBuffer()
 * \brief       Set up offline debug buffer
 *
 * This function shall be called to set up offline debug buffer.
 * This function will use the buffer created from user application to store the logging data.
 * \param[in]   pBuf pointer of user buffer 
 * \param[in]   size size of user buffer
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingSetBuffer(char *pBuf, uint32_t size);

/*!
 * \fn          sensor_error loggingStart(uint8_t DebugLevel)
 * \brief       Start logging all activities
 *
 * All device activity will be logged once this function is called.
 * This function shall be called after loggingOpen().
 * \param[in]   DebugLevel
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingStart(uint8_t DebugLevel);

/*!
 * \fn          sensor_error loggingStop()
 * \brief       Stop logging all activities
 *
 * All device activity will stop to be logged once this function is called.
 * This function shall be called after  loggingStart().
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingStop();

/*!
 * \fn          sensor_error logDebugMessageStart(const char* pFunctionName)
 * \brief       Write start section to log debug message
 * \param[in]   pFunctionName
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error logDebugMessageStart(const char* pFunctionName);

/*!
 * \fn          sensor_error logDebugMessageEnd()
 * \brief       Write end section to log debug message
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error logDebugMessageEnd();

/*!
 * \fn          sensor_error logErrorMessageStart(const char* pFunctionName)
 * \brief       Write start section to log error message
 * \param[in]   pFunctionName
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error logErrorMessageStart(const char* pFunctionName);

/*!
 * \fn          sensor_error logErrorMessageEnd()
 * \brief       Write end section to log error message
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error logErrorMessageEnd();

/*!
 * \fn          sensor_error loggingFunctionStart(const char* FunctionName, void **pFuncArguments)
 * \brief       log start of an API function
 * \param[in]   FunctionName
 * \param[in]   pFuncArguments
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingFunctionStart(const char* pFunctionName, void **pFuncArguments);

/*!
 * \fn          sensor_error loggingFunctionEnd(const char* pFunctionName, sensor_error ReturnedValue, void **pFuncArguments)
 * \brief       log end of an API function
 * \param[in]   pFunctionName
 * \param[in]   ReturnedValue
 * \param[in]   pFuncArguments
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingFunctionEnd(const char* pFunctionName, sensor_error ReturnedValue, void **pFuncArguments);

/*!
 * \fn          sensor_error loggingOutput(char* pBuffer)
 * \brief       Write messge bugger to actual output
 * \param[in]   pBuffer
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingOutput(char* pBuffer);

/*!
 * \fn          sensor_error loggingOfflineOutput()
 * \brief       Write messge bugger from offline buffer to actual output
 * \retval      SENSOR_ERROR_NONE : Success
 * \retval      "Other Error Code" : Failure
 */
sensor_error loggingOfflineOutput();

/* help compiling in C++  */
#ifdef __cplusplus
}
#endif   /*__cplusplus*/


#else

#define SENSOR_ERROR_LOG(...)    ;
#define DEBUG_LOG(...)           ;
#define LOG_FUNCTION_START(...)  ;
#define LOG_FUNCTION_END(...)    ;
#define loggingOpen()            SENSOR_ERROR_NONE
#define loggingClose()           SENSOR_ERROR_NONE
#define loggingStart(...)        SENSOR_ERROR_NONE
#define loggingStop()            SENSOR_ERROR_NONE
#define loggingGetSize(...)      SENSOR_ERROR_NONE
#define loggingReadBack(...)     SENSOR_ERROR_NONE
#define loggingOutput(...)       SENSOR_ERROR_NONE
#endif

#endif  /* DEBUG */


