/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

//! \defgroup	Debugging_Functions
#ifndef __KERNEL__
#include <string.h>
#else
#include <linux/kernel.h>
#include <linux/string.h>
#endif
#include "debug.h"
#include "definitions.h"
#include "als_driver.h"
#include "ranging_driver.h"
#include "i2c_log.h"

// Defines for internal use

#define DEBUG_NONE		0
#define DEBUG_BYTE		1
#define DEBUG_BOOL		1
#define DEBUG_UINT8		1
#define DEBUG_INT8		1
#define DEBUG_UINT16	2
#define DEBUG_INT16     2
#define DEBUG_UINT32	3
#define DEBUG_UINT64	4
#define DEBUG_FLOAT		5
#define DEBUG_DOUBLE    6
#define DEBUG_STRUCT	9



#ifdef DEBUG_ENABLE

#define DUMP_PARAM_STRUCT(ParamName, Param, ParamType)                                  dumpParameter(#ParamName, sizeof(ParamType), (uint8_t *)Param, DEBUG_STRUCT,   #ParamType)
#define DUMP_PARAM(ParamName, Param, ParamType)                                         dumpParameter(#ParamName, 1, (uint8_t *)Param, ParamType, NULL)
#define DUMP_PARAM_PTR(ParamName, Param, ParamSize, ParamType)                          dumpParameter(#ParamName, ParamSize, (uint8_t *)Param, ParamType, NULL)
#define DUMP_STRUCTPARAM(StructType, StructName, ParamName, ParamType)                  dumpParameter(#ParamName, 1, (uint8_t *)&(((StructType*)StructName)->ParamName), ParamType, NULL);
#define DUMP_STRUCTPARAM_PTR(StructType, StructName, ParamName, ParamSize, ParamType)   dumpParameter(#ParamName, ParamSize, (uint8_t *)(((StructType*)StructName)->ParamName), ParamType, NULL);
#define DUMP_STRUCTPARAM_ARRAY(StructType, StructName, ParamName, ParamSize, ParamType) dumpParameter(#ParamName, ParamSize, (uint8_t *)(&(((StructType*)StructName)->ParamName[0])),  ParamType, NULL);
#define DUMP_STRUCTPARAM_STRUCT(StructType, StructName, ParamName, ParamType)           dumpParameter(#ParamName, sizeof(ParamType), (uint8_t *)&(((StructType*)StructName)->ParamName), DEBUG_STRUCT, #ParamType);

#define DEBUG_TEST_STR_EQUALITY(Str1, Str2, UINT8_VAR) core_strCmpr(Str1, Str2, &UINT8_VAR);if(UINT8_VAR)



/* private functions for debugging */
sensor_error dumpParameters(const char* pFunctionName, void **pFuncArguments);
sensor_error dumpParameter(const char* pParamName, const uint32_t ParamLength, const uint8_t *pParamValues, const uint8_t ParamType, const char* pStructName);
sensor_error isApiCoreFunction(const char* pFunctionName, bool_t* isCoreFunc);
sensor_error core_strCmpr(const char* Str1, const char* Str2, bool_t *pEqual);
sensor_error logFunction(bool_t* pLogFunction, const char* pFunctionName, const bool_t IsCoreFunction, const bool_t FunctionStart);
sensor_error logFunctionParameters(bool_t* pLogFunctionParams, const char* pFunctionName, const bool_t IsCoreFunction, const bool_t FunctionStart);

/* Global variables used only by debug system */
#define  LINE_SIZE 1024
char     DebugBuffer[LINE_SIZE];
char *   pDebugBuffer=0;
char *   pOffLineBuffer=0;
uint32_t OffLineLogSize=0,CurLogSize=0;
bool_t	 DebugStarted=FALSE;

char	 DebugLogLevel = (char)NORMAL_LOG_LEVEL;
char     DebugMode = (char)DEBUG_ONLINE;
char	 DebugCurrentFunctionLevel = 0;


/**************************************************************/
/*	Debugging Functions										  */
/**************************************************************/
sensor_error loggingOpen()
{
	/* Debug String Line buffer */
	pDebugBuffer = &DebugBuffer[0];
	CurLogSize = 0;
	DebugCurrentFunctionLevel = 0;


	return SENSOR_ERROR_NONE;
}

sensor_error loggingClose()
{
	sensor_error Status = SENSOR_ERROR_NONE;
	if(DebugStarted){
		Status = loggingStop();
	}
	if(Status == SENSOR_ERROR_NONE){
		pDebugBuffer=0;
	}

	return Status;
}

sensor_error loggingSetMode (char mode)
{
	if (mode != DEBUG_ONLINE && mode != DEBUG_OFFLINE)
		return SENSOR_ERROR;
	DebugMode = mode;
	
	return SENSOR_ERROR_NONE;
}

sensor_error loggingSetBuffer (char *pBuf, uint32_t size)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	if (pBuf == 0 || size ==0)
		return SENSOR_ERROR;
	pOffLineBuffer = pBuf;
	OffLineLogSize = size;
	CurLogSize = 0;
	
	return Status;
}

sensor_error loggingStart(uint8_t DebugLevel)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t CurrentTime = 0;

	if(DebugMode==DEBUG_OFFLINE && pOffLineBuffer==0) {
		/* Offline mode but Log buffer not assigned yet, return error */
		debug1_print("loggingStart Error, OFFLINE mode, no Offline buffer provided yet!!!\n");
		return SENSOR_ERROR;
	}
	CurLogSize = 0;
	CurrentTime = timer_get_clock_time_msecs();
	
	if(pDebugBuffer==0) {
		/* Log buffer not assigned yet, create it first */
		Status = loggingOpen();
	}

	DebugLogLevel = DebugLevel;
	DebugStarted=TRUE;

	//if(Status == SENSOR_ERROR_NONE){
	//	Status = GetTimeStamp(&CurrentTime);
	//}
	if(Status == SENSOR_ERROR_NONE){
		DEBUG_WRITE_IN_LOG("<LoggingSession TimeStamp(ms)=\"%.8d\">\n", CurrentTime);
		DEBUG_WRITE_IN_LOG(" <DebugStart TimeStamp(ms)=\"%.8d\"/>\n", CurrentTime);
	}
	if (DebugLogLevel == I2C_LOG_LEVEL)
		i2c_log_start();
	else
		i2c_log_end();

	DebugCurrentFunctionLevel = 0;

	return SENSOR_ERROR_NONE;
}

sensor_error loggingStop()
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t CurrentTime = 0;

	CurrentTime = timer_get_clock_time_msecs();
	if(Status == SENSOR_ERROR_NONE){
		DEBUG_WRITE_IN_LOG(" <DebugStop TimeStamp(ms)=\"%.8d\"/>\n", CurrentTime);
		DEBUG_WRITE_IN_LOG("</LoggingSession>\n");
	}

	DebugStarted=FALSE;


	return SENSOR_ERROR_NONE;
}

sensor_error logDebugMessageStart(const char* pFunctionName)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t CurrentTime = 0;

	CurrentTime = timer_get_clock_time_msecs();
	if(Status==SENSOR_ERROR_NONE){
		DEBUG_WRITE_IN_LOG("<Log_message TimeStamp(ms)=\"%.8d\" API_Function_Name=\"%s\">\n",CurrentTime, pFunctionName);
	}

	return Status;
}

sensor_error logDebugMessageEnd()
{
	sensor_error Status = SENSOR_ERROR_NONE;

	DEBUG_WRITE_IN_LOG("</Log_message>\n");

	return Status;
}

sensor_error logErrorMessageStart(const char* pFunctionName)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t CurrentTime = 0;

	CurrentTime = timer_get_clock_time_msecs();
	if(Status==SENSOR_ERROR_NONE){
		DEBUG_WRITE_IN_LOG("<ERROR_MESSAGE TimeStamp(ms)=\"%.8d\" API_Function_Name=\"%s\">\n",CurrentTime, pFunctionName);

	}

	return Status;
}

sensor_error logErrorMessageEnd()
{
	sensor_error Status = SENSOR_ERROR_NONE;

	DEBUG_WRITE_IN_LOG("</ERROR_MESSAGE>\n");


	return Status;
}


sensor_error loggingFunctionStart(const char* pFunctionName, void **pFuncArguments)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t CurrentTime = 0;
	bool_t isCoreFunc = FALSE;
	bool_t logFunc;
	bool_t logFuncParams;

	if(DebugStarted){
        CurrentTime = timer_get_clock_time_msecs();
		if(Status==SENSOR_ERROR_NONE){
				/* Check if we are starting to log an API function */
				logFunction(&logFunc, pFunctionName, isCoreFunc, TRUE);
				logFunctionParameters(&logFuncParams, pFunctionName, isCoreFunc, TRUE);
				if(logFunc)
				{
					DEBUG_WRITE_IN_LOG("<API_Function>\n");
					DEBUG_WRITE_IN_LOG(" <Exec_Start TimeStamp(ms)=\"%.8d\">\n", CurrentTime);
					DEBUG_WRITE_IN_LOG("  <API_Function_Name>%s</API_Function_Name>\n", pFunctionName);

					if(logFuncParams)
					{
						DEBUG_WRITE_IN_LOG("  <API_Input_Arguments>\n");
						dumpParameters(pFunctionName, pFuncArguments);
						DEBUG_WRITE_IN_LOG("  </API_Input_Arguments>\n");
					}

					DEBUG_WRITE_IN_LOG("  </Exec_Start>\n");
				}

		}
	}
	return Status;
}


sensor_error loggingFunctionEnd(const char* pFunctionName, sensor_error ReturnedValue, void **pFuncArguments)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t      CurrentTime=0;
	bool_t 		  isCoreFunc=FALSE;
	bool_t 		  logFunc;
	bool_t 		  logFuncParams=FALSE;

	if(DebugStarted){
		CurrentTime = timer_get_clock_time_msecs();

	if(Status==SENSOR_ERROR_NONE){

				logFunction(&logFunc, pFunctionName, isCoreFunc, FALSE);
				logFunctionParameters(&logFuncParams, pFunctionName, isCoreFunc, FALSE);

				if(logFunc)
				{
					DEBUG_WRITE_IN_LOG(" <Exec_End TimeStamp(ms)=\"%.8d\"/>\n", CurrentTime);
					DEBUG_WRITE_IN_LOG("  <API_Function_Name>%s</API_Function_Name>\n", pFunctionName);

					if(logFuncParams)
					{
						DEBUG_WRITE_IN_LOG("  <API_Output_Arguments>\n");
						dumpParameters(pFunctionName, pFuncArguments);
						DEBUG_WRITE_IN_LOG("  </API_Output_Arguments>\n");
					}

					DEBUG_WRITE_IN_LOG("  <Returned_Value>0x%x</Returned_Value>\n", ReturnedValue);
				}
				if(logFunc)
				{
					DEBUG_WRITE_IN_LOG("</API_Function>\n");
				}
		}
	}
	return Status;
}

sensor_error loggingOutput(char* pBuffer)
{
		//check the mode
		if (DebugMode == DEBUG_ONLINE)
		{
			debug1_print(pBuffer);
    }
		else
		{
				if (CurLogSize + strlen(pBuffer) < OffLineLogSize)
				{
					sprintf(pOffLineBuffer+CurLogSize,pBuffer);
					CurLogSize += strlen(pBuffer);
				}
		}		
		
		return SENSOR_ERROR_NONE;
}

sensor_error loggingOfflineOutput(void)
{
		if (DebugMode == DEBUG_OFFLINE)
		{
				uint32_t i=0,size=0;
				while (i <CurLogSize)
				{
						if ((CurLogSize-i) > LINE_SIZE)
							size = LINE_SIZE;
						else
							size = (CurLogSize-i);
						strncpy(pDebugBuffer,(pOffLineBuffer+i), size);
						debug1_print(pDebugBuffer);
						i+=size;
				}
			
		}	
		return SENSOR_ERROR_NONE;			
}

sensor_error core_strCmpr(const char* Str1, const char* Str2, bool_t *pEqual)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint16_t Str1Length = 0;
	uint16_t Str2Length = 0;
	uint16_t Index;

	*pEqual=FALSE;
	Index=0;
	while(((uint8_t)(*(Str1+Index)))!=0)
	{
		Index=Index+1;
	}
	Str1Length=Index;

	Index=0;
	while(((uint8_t)(*(Str2+Index)))!=0)
	{
		Index=Index+1;
	}
	Str2Length=Index;

	if(Str1Length != Str2Length){
		*pEqual=FALSE;
		return Status;
		}

	for(Index=0;Index<Str1Length;Index++){
		if(((uint8_t)(*(Str1+Index)))!=((uint8_t)(*(Str2+Index)))){
			*pEqual=FALSE;
			return Status;
	}
	}

	*pEqual=TRUE;
	return Status;
}


sensor_error isApiCoreFunction(const char* pFunctionName, bool_t* isCoreFunc)
{
	sensor_error Status = SENSOR_ERROR_NONE;

	char corefunctionRoot[]="core";
	char functionRoot[13];
	uint8_t	i;

	for(i=0; ((i<12)&&(pFunctionName[i]!=0)); i=i+1)
	{
		functionRoot[i]=pFunctionName[i];
	}
	functionRoot[i]=0;

	/* Test Equality */
	Status = core_strCmpr(functionRoot, corefunctionRoot, isCoreFunc);

	return Status;
}


sensor_error logFunction(bool_t* pLogFunction, const char* pFunctionName, const bool_t IsCoreFunction, const bool_t FunctionStart)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	const uint8_t NbOfFunctions=36;
	//#heavy load APIs only output for verbos mode
	char *functions[36] = { //has to match NbOfFunctions.
						"getApiVersionNumber",
						"getRangeMeasurement",
						"getAlsMeasurement",
						"readRegister",
						"range_set_systemMode",
						"range_get_result",
						"range_get_result_status",
						"range_get_device_ready",
						"range_get_result_error_codes",
						"range_get_max_convergence_time",
						"range_set_system_interrupt_clear",
						"range_get_result_interrupt_status_gpio",
						"als_get_lux",
						"als_get_result",
						"als_set_system_interrupt_config_gpio",
						"als_set_system_interrupt_clear",
						"als_get_result_interrupt_status_gpio",
						"common_set_gpio0_mode",
						"common_get_gpio0_mode",
						"common_set_gpio0_select",
						"common_get_gpio0_select",
						"common_set_gpio0_polarity",
						"common_get_gpio0_polarity",
						"common_set_gpio1_mode",
						"common_set_gpio1_select",
						"common_get_gpio1_select",
						"common_set_gpio1_polarity",
						"common_get_gpio1_polarity",
						"common_get_history_buffer_clear",
						"common_set_system_interrupt_config_gpio_ranging",
						"common_set_system_interrupt_clear_ranging",
						"common_set_system_interrupt_clear_error",
						"common_get_range_result_interrupt_status_gpio",
						"common_get_error_result_interrupt_status_gpio",
						"er_get_result",
						"er_set_scaler"
					   };
	uint8_t i=0;
	bool_t StrEquality;
	/* Depending on Log level, we will log only:
	 * API UI calls in MINIMUM_LOG_LEVEL
	 * API UI and core calls in NORMAL_LOG_LEVEL and VERBOSE_LOG_LEVEL */
	if (DebugLogLevel==I2C_LOG_LEVEL)
	{
			*pLogFunction=FALSE;
			return Status;
	}
	if (DebugLogLevel==VERBOSE_LOG_LEVEL)
	{
			*pLogFunction=TRUE;
			return Status;
	}
	for(i=0;i<NbOfFunctions;i++)
	{
		core_strCmpr(pFunctionName, functions[i], &StrEquality);
		if(StrEquality==TRUE)
		{
				*pLogFunction=FALSE;
				break;
		}
	}

	return Status;
}

sensor_error logFunctionParameters(bool_t* pLogFunctionParams, const char* pFunctionName, const bool_t IsCoreFunction, const bool_t FunctionStart)
{
	sensor_error Status = SENSOR_ERROR_NONE;

	//#heavy load APIs only output for verbos mode
	char *functions[1] = {
				"to_define_later"
					   };
	uint8_t NbOfFunctions=0;
	uint8_t i;
	bool_t DoLog;
	bool_t StrEquality;


	/* Depending on Log level, we will log only:
	 * API UI calls parameters in ILP0013_MINIMUM_LOG_LEVEL and ILP0013_NORMAL_LOG_LEVEL
	 * API UI and core parameters in ILP0013_VERBOSE_LOG_LEVEL */
	switch(DebugLogLevel)
	{
	case MINIMUM_LOG_LEVEL:
		if(IsCoreFunction)
			*pLogFunctionParams=FALSE;
		else
		{
			if(FunctionStart)
				DoLog = TRUE;
			else
				DoLog = FALSE;
			for(i=0;i<NbOfFunctions;i++)
			{
				core_strCmpr(pFunctionName, functions[i], &StrEquality);
				if(StrEquality==TRUE)
				{
					DoLog = (bool_t)!DoLog;
					break;
				}
			}

			*pLogFunctionParams=DoLog;
		}
		break;
	case NORMAL_LOG_LEVEL:
		if(IsCoreFunction)
			*pLogFunctionParams=FALSE;
		else
			*pLogFunctionParams=TRUE;
		break;
	case VERBOSE_LOG_LEVEL:
		*pLogFunctionParams=TRUE;
		break;
	case I2C_LOG_LEVEL:
		*pLogFunctionParams=FALSE;
		break;
	default:
		*pLogFunctionParams=TRUE;
		break;
	}

	return Status;
}

sensor_error dumpParameters(const char* pFunctionName, void **pFuncArguments)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	bool_t FuncStrEqual;

    /* API FUNCTIONS    */
    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_history_buffer_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_snr_thresh", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(snrThresh, *(pFuncArguments+1), DEBUG_FLOAT);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_system_fresh_out_of_reset", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_lower_limit", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_i2c_pad_voltage", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio1_polarity", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(polarity, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_get_scaler", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_system_interrupt_config_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_vhv_repeat_rate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(VHV_repeat_rate, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "decodeFrom4_4_Format", FuncStrEqual){
        DUMP_PARAM(value, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_scaler", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_crosstalk_compensation_rate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(crosstalk_compensation_rate, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_interMeasurement_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(intermeasurement_period, *(pFuncArguments+1), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_interMeasurement_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(intermeasurement_period, *(pFuncArguments+1), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_system_interrupt_config_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(Ranging_GPIO_interrupt_config, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_ece_factor", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(ECE_Factor_M, *(pFuncArguments+1), DEBUG_UINT32);
        DUMP_PARAM(ECE_Factor_D, *(pFuncArguments+2), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio0_polarity", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio1_select", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "get_als_measurement", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM_STRUCT(pAlsData, *(pFuncArguments+1), sensor_AlsData);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_system_interrupt_config_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(ALS_GPIO_interrupt_config, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "Range_Set_Thresholds", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(low_threshold, *(pFuncArguments+1), DEBUG_UINT32);
        DUMP_PARAM(high_threshold, *(pFuncArguments+2), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "initialise", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_interMeasurement_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "get_range_measurement", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM_STRUCT(pRangeData, *(pFuncArguments+1), sensor_RangeData);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_early_convergence_estimate_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_get_lower_limit", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_history_buffer_mode_enable", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "encodeTo9_7_Format", FuncStrEqual){
        DUMP_PARAM(value, *(pFuncArguments+0), DEBUG_FLOAT);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_early_convergence_estimate_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_systemMode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "decodeFrom9_7_Format", FuncStrEqual){
        DUMP_PARAM(value, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_ignore_valid_height", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(ignore_valid_height, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_system_interrupt_clear", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_vhv_recalibrate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_device_ready", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_initialise", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_low_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_set_part2Part_range_offset", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(part_to_part_range_offset, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_system_interrupt_config_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_i2c_base_address", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(i2c_base_address, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "roundFloatToInt", FuncStrEqual){
        DUMP_PARAM(floatVal, *(pFuncArguments+0), DEBUG_FLOAT);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_high_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_set_static_config", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "unPackBytes", FuncStrEqual){
        DUMP_PARAM(data, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM_PTR(byteArray, *(pFuncArguments+1), 1, DEBUG_UINT8);
        DUMP_PARAM(size, *(pFuncArguments+2), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_history_buffer_enable", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_max_convergence_time", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_range_get_part2Part_range_offset", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_dynamic_config", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_system_interrupt_clear", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_result_status", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_result", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_range_check_enables", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_integration_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(integration_period, *(pFuncArguments+1), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_result", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_clear_interleaved_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_device_ready", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_interMeasurement_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio0_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_system_health_check", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_result_interrupt_status_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_history_buffer_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(history_buffer_mode, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_ece_factor", FuncStrEqual){
        DUMP_PARAM_PTR(pece_factor_m, *(pFuncArguments+0), 1, DEBUG_UINT32);
        DUMP_PARAM_PTR(pece_factor_d, *(pFuncArguments+1), 1, DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_analogue_gain", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(light_analogue_gain, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio1_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_system_group_parameter_hold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_signal_rate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_get_result", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_clear_system_fresh_out_of_reset", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "Get_Range_History_Buffer", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_max_convergence_time", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(max_convergence_time, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_scaler", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(scaler, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_static_config", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_vhv_repeat_rate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_crosstalk_compensation_rate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "get_version", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM_PTR(pVersionStr, *(pFuncArguments+1), 1, DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_snr_thresh", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_system_fatal_error_code", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_systemMode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_high_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio1_select", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(select, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_thresholds", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(low_threshold, *(pFuncArguments+1), DEBUG_UINT16);
        DUMP_PARAM(high_threshold, *(pFuncArguments+2), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "i2c_initialise", FuncStrEqual){
        DUMP_PARAM(addr, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_low_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio1_polarity", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_upper_limit", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_history_buffer_mode_enable", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_crosstalk_compensation_range", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_high_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(threshold, *(pFuncArguments+1), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_crosstalk_valid_height", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_history_buffer_clear", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_full_result", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_interleaved_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "Get_ALS_History_Buffer", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_analogue_gain", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_clear_system_group_parameter_hold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "start_extended_ranging", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_set_scaler", FuncStrEqual){
        DUMP_PARAM(scaler, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(device_base_address, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_system_fatal_error_status", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_result_interrupt_status_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_range_check_enables", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(range_check_enables, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "encodeTo4_4_Format", FuncStrEqual){
        DUMP_PARAM(value, *(pFuncArguments+0), DEBUG_FLOAT);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_i2c_pad_voltage", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(pad_voltage, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio0_polarity", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(polarity, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_gpio0_select", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "packBytes", FuncStrEqual){
        DUMP_PARAM_PTR(byteArray, *(pFuncArguments+0), 1, DEBUG_UINT8);
        DUMP_PARAM(size, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_integration_period", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_vernier_ave_total_time", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_result_status", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio0_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_system_interrupt_clear_error", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "start_als", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_systemMode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_range_ignore_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_error_result_interrupt_status_gpio", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_lux", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_systemMode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_low_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(threshold, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio0_select", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(select, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_history_buffer_enable", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(history_buffer_enable, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_get_result_error_codes", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_crosstalk_valid_height", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(crosstalk_valid_height, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_system_mode_gpio0", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT8);
        DUMP_PARAM(select, *(pFuncArguments+2), DEBUG_UINT8);
        DUMP_PARAM(polarity, *(pFuncArguments+3), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_system_mode_gpio1", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT8);
        DUMP_PARAM(select, *(pFuncArguments+2), DEBUG_UINT8);
        DUMP_PARAM(polarity, *(pFuncArguments+3), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_dynamic_config", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "i2c_read_uint32", FuncStrEqual){
        DUMP_PARAM(reg, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM(baseAddr, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_low_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(threshold, *(pFuncArguments+1), DEBUG_UINT16);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "i2c_write", FuncStrEqual){
        DUMP_PARAM(reg, *(pFuncArguments+0), DEBUG_UINT32);
        DUMP_PARAM_PTR(data, *(pFuncArguments+1), 1, DEBUG_UINT8);
        DUMP_PARAM(size, *(pFuncArguments+2), DEBUG_UINT32);
        DUMP_PARAM(baseAddr, *(pFuncArguments+3), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "als_set_interleaved_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_converg_ctrl_rtn_thresh_fine", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_emitter_block_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(emitter_block_threshold, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_vhv_recalibrate", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(VHV_Recalibrate, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_crosstalk_compensation_range", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(crosstalk_compensation_range, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "get_vendor", FuncStrEqual){
        DUMP_PARAM_PTR(pVendorStr, *(pFuncArguments+0), 1, DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_result_error_codes", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_history_buffer_clear", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_ignore_valid_height", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_range_ignore_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(range_ignore_threshold, *(pFuncArguments+1), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_get_emitter_block_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_set_gpio1_mode", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(mode, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "common_get_identification", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "range_set_high_threshold", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
        DUMP_PARAM(threshold, *(pFuncArguments+1), DEBUG_UINT8);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "start_ranging", FuncStrEqual){
        DUMP_PARAM(id, *(pFuncArguments+0), DEBUG_UINT32);
    }

    DEBUG_TEST_STR_EQUALITY(pFunctionName, "er_get_upper_limit", FuncStrEqual){
        DUMP_PARAM(device_base_address, *(pFuncArguments+0), DEBUG_UINT8);
    }

    /* API CORE FUNCTIONS    */
	/* END API FUNCTIONS */
	return Status;
}

sensor_error dumpParameter(const char* pParamName, const uint32_t ParamLength, const uint8_t *pParamValues, const uint8_t ParamType, const char* pStructName)
{
	sensor_error Status = SENSOR_ERROR_NONE;
	uint32_t I;
	//bool_t ParamStrEqual;

    DEBUG_WRITE_IN_LOG("   <%s>",pParamName);

    if(ParamType==DEBUG_STRUCT){
        /* DUMP STRUCTURES */
        /* END DUMP STRUCTURES */
		    /* END DUMP STRUCTURES */
	} else  {
		for(I=0; I<ParamLength; I++){
			switch(ParamType)
			{
				case DEBUG_NONE:
					/* SHOULD NEVER ENTER HERE */
					break;
				case DEBUG_BYTE:
					DEBUG_WRITE_IN_LOG("0x%02x",(uint8_t)*(pParamValues+I));
					break;
				case DEBUG_UINT16:
					DEBUG_WRITE_IN_LOG("0x%04x",*(uint16_t*)(pParamValues+(I*2)));
					break;
				case DEBUG_UINT32:
					DEBUG_WRITE_IN_LOG("0x%08x",*(uint32_t*)(pParamValues+(I*4)));
					break;
				case DEBUG_UINT64:
					DEBUG_WRITE_IN_LOG("0x%016x",*(uint32_t*)(pParamValues+(I*8)));
					break;
				case DEBUG_FLOAT:
					DEBUG_WRITE_IN_LOG("0x%08x",*(uint32_t*)(pParamValues+(I*4)));
					break;
				case DEBUG_DOUBLE:
					DEBUG_WRITE_IN_LOG("0x%08x",*(uint32_t*)(pParamValues+(I*8)));
					break;
				default:
					break;
			}
			/*
			if(I<(ParamLength-1))
				DEBUG_WRITE_IN_LOG(", ");
			*/
		}

	}

	DEBUG_WRITE_IN_LOG("</%s>\n",pParamName);

	return Status;
}

#endif

