/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file range_upscaling_driver.h
 *\brief Application-level functions used for configuration and operation during upscaled ranging.
 */


#ifndef EXT_RANGING_DRIVER
#define EXT_RANGING_DRIVER

//-----------------------------------------------------------------------------
// module imports
//-----------------------------------------------------------------------------

// ST libraries
#include "definitions.h"

//-----------------------------------------------------------------------------
// type definitions
//-----------------------------------------------------------------------------
typedef enum { Scale1X = 1, Scale2X = 2, Scale3X = 3 } ExtRangeScaler;

//-----------------------------------------------------------------------------
// method definitions
//-----------------------------------------------------------------------------

/*!
 *\brief Device setup for Upscaled Ranging operations.
 *
 * Device setup for extended-range operations. To apply these settings the operation bit (bit 0) in the SYSRANGE__START & SYSALS__START Registers must be cleared.
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error er_set_static_config(uint8_t device_base_address);

/*!
 *\brief Set the Part-to-Part Range Offset
 *
 * Set part-to-part range offset in the sysrange_part_to_part_range_offset register.
 *
 *\param[in] device_base_address
 *\param[in] part_to_part_range_offset
 *\retval sensor_error
 */
sensor_error er_set_part2Part_range_offset(uint8_t device_base_address, uint8_t part_to_part_range_offset);

/*!
 *\brief Report the part-to-part range offset.
 *
 * Report the part-to-part range offset from sysrange_part_to_part_range_offset register.
 *
 *\param[in] device_base_address
 *\retval Byte-wide, integer
 */
uint8_t er_range_get_part2Part_range_offset(uint8_t device_base_address);

/*!
 *\brief Report result from last completed upscaled ranging operation.
 *\param[in] device_base_address
 *\retval 32-bit integer
 */
uint32_t er_get_result(uint8_t device_base_address);

/*!
 *\brief Set the extended range scaling to either 2X or 3X scaling. An increased scaler
 *\      will result in a greater measurement distance but more coarse resolution.
 *\param[in] scaler
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error er_set_scaler(uint8_t scaler, uint8_t device_base_address);

/*!
 *\brief Reports the current scaler setting.
 *\param[in] device_base_address
 *\retval Byte-wide, integer
 */
uint8_t er_get_scaler(uint8_t device_base_address);


/**
 * @brief Function to return the Maximum range reported by the
 *        sensor.
 *\param[in] device_base_address
 *\retval 32-bit integer
 */
uint32_t er_get_upper_limit(uint8_t device_base_address);

/**
 * @brief Function to return the Minimum range reported by the
 * sensor.
 *\param[in] device_base_address
 *\retval 32-bit integer
 */
uint32_t er_get_lower_limit(uint8_t device_base_address);

#endif



