/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*
  '''
  Application-level methods used for generic, system or identification operations.
  '''
*/

//-----------------------------------------------------------------------------
// module imports
//-----------------------------------------------------------------------------
#include "debug.h"
#include "common_driver.h"
#include "ranging_driver.h"

//-----------------------------------------------------------------------------
// global variable declarations
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// method definitions
//-----------------------------------------------------------------------------

sensor_error common_initialise(uint8_t device_base_address)
{
	
    LOG_FUNCTION_START((void*)&device_base_address);

    i2c_initialise(device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error common_set_static_config(uint8_t device_base_address)
{
    int8_t reset;
	
    LOG_FUNCTION_START((void*)&device_base_address);
    reset = i2c_read_byte(0x016, device_base_address);
    if (reset==1)
    {
        // check to see has it be Initialised already
        //REGISTER_TUNING_SR03_270514_CustomerView.txt

        // Mandatory : private registers
        i2c_write_byte(0x0207, 0x01, device_base_address);
        i2c_write_byte(0x0208, 0x01, device_base_address);
        i2c_write_byte(0x0096, 0x00, device_base_address);
        i2c_write_byte(0x0097, 0xfd, device_base_address);
        i2c_write_byte(0x00e3, 0x00, device_base_address);
        i2c_write_byte(0x00e4, 0x04, device_base_address);
        i2c_write_byte(0x00e5, 0x02, device_base_address);
        i2c_write_byte(0x00e6, 0x01, device_base_address);
        i2c_write_byte(0x00e7, 0x03, device_base_address);
        i2c_write_byte(0x00f5, 0x02, device_base_address);
        i2c_write_byte(0x00d9, 0x05, device_base_address);
        i2c_write_byte(0x00db, 0xce, device_base_address);
        i2c_write_byte(0x00dc, 0x03, device_base_address);
        i2c_write_byte(0x00dd, 0xf8, device_base_address);
        i2c_write_byte(0x009f, 0x00, device_base_address);
        i2c_write_byte(0x00a3, 0x3c, device_base_address);
        i2c_write_byte(0x00b7, 0x00, device_base_address);
        i2c_write_byte(0x00bb, 0x3c, device_base_address);
        i2c_write_byte(0x00b2, 0x09, device_base_address);
        i2c_write_byte(0x00ca, 0x09, device_base_address);
        i2c_write_byte(0x0198, 0x01, device_base_address);
        i2c_write_byte(0x01b0, 0x17, device_base_address);
        i2c_write_byte(0x01ad, 0x00, device_base_address);
        i2c_write_byte(0x00ff, 0x05, device_base_address);
        i2c_write_byte(0x0100, 0x05, device_base_address);
        i2c_write_byte(0x0199, 0x05, device_base_address);
        i2c_write_byte(0x01a6, 0x1b, device_base_address);
        i2c_write_byte(0x01ac, 0x3e, device_base_address);
        i2c_write_byte(0x01a7, 0x1f, device_base_address);
        i2c_write_byte(0x0030, 0x00, device_base_address);

        // Recommended : Public registers - See data sheet for more detail
        i2c_write_byte(0x0011, 0x10, device_base_address); // Enables polling for New Sample ready when measurement completes
        i2c_write_byte(0x010a, 0x30, device_base_address); // Set the averaging sample period (compromise between lower noise and increased execution time)
        i2c_write_byte(0x003f, 0x46, device_base_address); // Sets the light and dark gain (upper nibble). Dark gain should not be changed.
        i2c_write_byte(0x0031, 0xFF, device_base_address); // sets the # of range measurements after which auto calibration of system is performed 
        i2c_write_byte(0x0040, 0x63, device_base_address); // Set ALS integration time to 100ms
        i2c_write_byte(0x002e, 0x01, device_base_address); // perform a single temperature calibration of the ranging sensor

        // Optional: Public registers - See data sheet for more detail
        i2c_write_byte(0x001b, 0x09, device_base_address); // Set default ranging inter-measurement period to 100ms
        i2c_write_byte(0x003e, 0x31, device_base_address); // Set default ALS inter-measurement period to 500ms
        i2c_write_byte(0x0014, 0x24, device_base_address); // Configures interrupt on New sample ready

        // extra stuff
        i2c_write_byte(0x016, 0x00, device_base_address); // change fresh out of set status to 0

    }

    // VHV automatically run on parts that are NVM-programmed, ie customer parts!

    range_set_max_convergence_time(device_base_address, 50); // Calculate ece value on initialisation (use max conv)
    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error common_get_identification(uint8_t device_base_address)
{
/*
  return_list = []

  # I2C base address is 8-bit indexing
  I2C_base_address = I2C_ReadReg_CCI(I2C_SLAVE_DEVICE_ADDRESS, I2C_ADDR);
  I2C_base_address *= 2 # convert from 7-bit to 8-bit address

  # reading from indices 0x00-0x0B
  regs = Utilities.device.comms['read'](I2C_base_address, 0x00, 0x0C)

  identification_model_id = regs[0x00]
  identification_model_rev_major = regs[0x01]
  identification_model_rev_minor = regs[0x02]
  identification_module_rev_major = regs[0x03]
  identification_module_rev_minor = regs[0x04]
  identification_nvm_revision_id = regs[0x05] & 0x0F
  identification_mask_revision_id = (regs[0x05] & 0xF0) >> 4
  identification_month = regs[0x06] & 0x0F
  identification_year = (regs[0x06] & 0xF0) >> 4
  identification_phase = regs[0x07] & 0x0F
  identification_day = (regs[0x07] & 0xF0) >> 4
  identification_time = (regs[0x08] << 8) + regs[0x09]
  identification_code = regs[0x0A]
  firmware_revision_id = regs[0x0B]

  return_list.append( identification_model_id )
  return_list.append( identification_model_rev_major )
  return_list.append( identification_model_rev_minor )
  return_list.append( identification_module_rev_major )
  return_list.append( identification_module_rev_minor )
  return_list.append( identification_nvm_revision_id )
  return_list.append( identification_mask_revision_id )
  return_list.append( identification_month )
  return_list.append( identification_year )
  return_list.append( identification_phase )
  return_list.append( identification_day )
  return_list.append( identification_time )
  return_list.append( identification_code )
  return_list.append( firmware_revision_id )
  return return_list
*/
    return SENSOR_ERROR_NONE;
}


sensor_error common_set_i2c_base_address(uint8_t device_base_address, uint32_t new_i2c_base_address)
{
    //# reduce 8-bit address to a 7-bit address
    LOG_FUNCTION_START((void*)&device_base_address,(void*)&new_i2c_base_address);

    i2c_write_byte(I2C_SLAVE_DEVICE_ADDRESS, (new_i2c_base_address >> 1), device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error common_set_i2c_pad_voltage(uint8_t device_base_address, uint8_t pad_voltage)
{
    uint8_t padConfigReg;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&pad_voltage);

    padConfigReg = i2c_read_byte(PAD_I2C_CONFIG, device_base_address);
    if (pad_voltage == I2C_1v2_PAD_VOLTAGE)
    {
        i2c_write_byte(PAD_I2C_CONFIG, (padConfigReg & 0xFE), device_base_address); // clear bit 0
    }

    if (pad_voltage == I2C_2v8_PAD_VOLTAGE)
    {
        i2c_write_byte(PAD_I2C_CONFIG, (padConfigReg & 0x01), device_base_address); // set bit 0
    }

	LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

uint8_t common_get_i2c_pad_voltage(uint8_t device_base_address)
{
    uint8_t ret=0;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (!(i2c_read_byte(PAD_I2C_CONFIG, device_base_address) & 0xFE)) // test that bit 0 is clear!
    {
        ret = I2C_1v2_PAD_VOLTAGE;
    }
    else if (i2c_read_byte(PAD_I2C_CONFIG, device_base_address) & 0x01) // test that bit 0 is set!
    {
        ret = I2C_2v8_PAD_VOLTAGE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_system_mode_gpio0(uint8_t device_base_address, uint8_t mode, uint8_t select, uint8_t polarity)
{
    uint8_t gpioModeReg;
    uint8_t gpio_select;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&mode, (void*)&select, (void*)&polarity);

    switch (select)
    {
        case GPIOx_SELECT_OFF :
        case GPIOx_SELECT_MEASURE_READY :
        case GPIOx_SELECT_DISABLED :
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT :
            gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
            gpioModeReg = gpioModeReg & 0x21; // preserve only the mode & polarity bits from the register contents!
            gpio_select = select << 1; // move up 1 bit, to make mask for bits 1-4

            i2c_write_byte(SYSTEM_MODE_GPIO0,
                                   (gpioModeReg | gpio_select),
                                   device_base_address);
            break;
        default :
            ret = COMMON_INVALID_PARAMS;
            break;
    }

    if ((mode == GPIOx_MODE_SELECT_RANGING) || (mode == GPIOx_MODE_SELECT_ALS))
    {
        gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
        i2c_write_byte(SYSTEM_MODE_GPIO0,
                               (gpioModeReg | mode),
                               device_base_address);
    }
    else
    {
        ret = COMMON_INVALID_PARAMS;
    }

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
    if (polarity == GPIOx_POLARITY_SELECT_OFF)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO0,
                               (gpioModeReg & GPIOx_POLARITY_SELECT_CLEARED),
                               device_base_address);
    }
    else if (polarity == GPIOx_POLARITY_SELECT_ON)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO0,
                               (gpioModeReg | GPIOx_POLARITY_SELECT),
                               device_base_address);
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio0_mode(uint8_t device_base_address, uint8_t mode)
{
    uint8_t gpioModeReg;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&mode);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
    if ((mode == GPIOx_MODE_SELECT_RANGING) || (mode == GPIOx_MODE_SELECT_ALS))
    {
        i2c_write_byte(SYSTEM_MODE_GPIO0, (gpioModeReg | mode), device_base_address);
    }
    else
    {
        ret = COMMON_INVALID_PARAMS;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_gpio0_mode(uint8_t device_base_address)
{
    uint8_t ret=0;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address) & GPIOx_MODE_SELECT)
    {
        ret = 1;
    }
    else
    {
        ret = 0;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio0_select(uint8_t device_base_address, uint8_t select)
{
    uint8_t gpio_select;
    uint8_t gpioModeReg;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&select);

    switch (select)
    {
        case GPIOx_SELECT_OFF:
        case GPIOx_SELECT_MEASURE_READY:
        case GPIOx_SELECT_THRESHOLD_OUTPUT:
        case GPIOx_SELECT_BLANK_IN:
        case GPIOx_SELECT_BLANK_OUT:
        case GPIOx_SELECT_START_STOP:
        case GPIOx_SELECT_DISABLED:
        case GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT:
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT:
            gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
            gpio_select = select << 1; // move up 1 bit, to make mask for bits 1-4
            i2c_write_byte(SYSTEM_MODE_GPIO0, (gpioModeReg | gpio_select), device_base_address);
            break;
        default :
            ret = COMMON_INVALID_PARAMS;
            break;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_gpio0_select(uint8_t device_base_address)
{
    uint8_t gpioModeReg;
    uint8_t gpio_select;
    uint8_t ret=0;

    LOG_FUNCTION_START((void*)&device_base_address);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
    gpio_select = (gpioModeReg & 0x1E) >> 1; // mask only bits 1-4
    switch (gpio_select)
    {
        case GPIOx_SELECT_OFF:
            ret = GPIOx_SELECT_OFF;
            break;
        case GPIOx_SELECT_MEASURE_READY:
            ret =  GPIOx_SELECT_MEASURE_READY;
            break;
        case GPIOx_SELECT_THRESHOLD_OUTPUT:
            ret =  GPIOx_SELECT_THRESHOLD_OUTPUT;
            break;
        case GPIOx_SELECT_BLANK_IN:
            ret =  GPIOx_SELECT_BLANK_IN;
            break;
        case GPIOx_SELECT_BLANK_OUT:
            ret =  GPIOx_SELECT_BLANK_OUT;
            break;
        case GPIOx_SELECT_START_STOP:
            ret =  GPIOx_SELECT_START_STOP;
            break;
        case GPIOx_SELECT_DISABLED:
            ret =  GPIOx_SELECT_DISABLED;
            break;
        case GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT:
            ret =  GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT;
            break;
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT:
            ret =  GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT;
            break;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio0_polarity(uint8_t device_base_address, uint8_t polarity)
{
    uint8_t gpioModeReg;
    sensor_error ret = SENSOR_ERROR_NONE;
	
    LOG_FUNCTION_START((void*)&device_base_address,(void*) &polarity);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address);
    if (polarity == GPIOx_POLARITY_SELECT_OFF)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO0,
                               (gpioModeReg & GPIOx_POLARITY_SELECT_CLEARED),
                               device_base_address);
    }
    else if (polarity == GPIOx_POLARITY_SELECT_ON)
        i2c_write_byte(SYSTEM_MODE_GPIO0, (gpioModeReg | GPIOx_POLARITY_SELECT), device_base_address);
    else
        ret = COMMON_INVALID_PARAMS;

	LOG_FUNCTION_END(ret);
		
    return ret;
}

bool_t common_get_gpio0_polarity(uint8_t device_base_address)
{
		bool_t ret=FALSE;
	
    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_MODE_GPIO0, device_base_address) & GPIOx_POLARITY_SELECT)
        ret = TRUE;

	LOG_FUNCTION_END(ret);
		
    return ret;
}

sensor_error common_set_system_mode_gpio1(uint8_t device_base_address, uint8_t mode, uint8_t select, uint8_t polarity)
{
    uint8_t gpioModeReg;
    uint8_t gpio_select;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&mode,(void*)&select,(void*)&polarity);

    switch (select)
    {
        case GPIOx_SELECT_OFF :
        case GPIOx_SELECT_MEASURE_READY :
        case GPIOx_SELECT_DISABLED :
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT :
            gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);
            gpioModeReg = gpioModeReg & 0x21; // preserve only the mode & polarity bits from the register contents!
            gpio_select = select << 1; // move up 1 bit, to make mask for bits 1-4
            i2c_write_byte(SYSTEM_MODE_GPIO1, (gpioModeReg | gpio_select), device_base_address);
            break;
        default :
            ret = COMMON_INVALID_PARAMS;
            break;
    }

    if ((mode == GPIOx_MODE_SELECT_RANGING) || (mode == GPIOx_MODE_SELECT_ALS))
    {
        gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg | mode),
                               device_base_address);
    }
    else
    {
        ret = COMMON_INVALID_PARAMS;
    }

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);

    if (polarity == GPIOx_POLARITY_SELECT_OFF)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg & GPIOx_POLARITY_SELECT_CLEARED),
                               device_base_address);
    }

    if (polarity == GPIOx_POLARITY_SELECT_ON)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg | GPIOx_POLARITY_SELECT),
                               device_base_address);
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio1_mode(uint8_t device_base_address, uint8_t mode)
{
    uint8_t gpioModeReg;
    sensor_error ret = COMMON_INVALID_PARAMS;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&mode);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);

    if ((mode == GPIOx_MODE_SELECT_RANGING) || (mode == GPIOx_MODE_SELECT_ALS))
    {
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg | mode),
                               device_base_address);
        ret = SENSOR_ERROR_NONE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_gpio1_mode(uint8_t device_base_address)
{
    uint8_t ret = 0;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address) & GPIOx_MODE_SELECT)
    {
        ret = 1;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio1_select(uint8_t device_base_address, uint8_t select)
{
    uint8_t gpio_select;
    uint8_t gpioModeReg;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&select);

    switch (select)
    {
        case GPIOx_SELECT_OFF:
        case GPIOx_SELECT_MEASURE_READY:
        case GPIOx_SELECT_THRESHOLD_OUTPUT:
        case GPIOx_SELECT_BLANK_IN:
        case GPIOx_SELECT_BLANK_OUT:
        case GPIOx_SELECT_START_STOP:
        case GPIOx_SELECT_DISABLED:
        case GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT:
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT:
            gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);
            gpio_select = select << 1; // move up 1 bit, to make mask for bits 1-4
            i2c_write_byte(SYSTEM_MODE_GPIO1,
                                   (gpioModeReg | gpio_select),
                                   device_base_address);
            break;
        default:
            ret = COMMON_INVALID_PARAMS;
            break;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_gpio1_select(uint8_t device_base_address)
{
    uint8_t gpio_select;
    uint8_t gpioModeReg;
    uint8_t ret=0;

    LOG_FUNCTION_START((void*)&device_base_address);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);

    gpio_select = (gpioModeReg & 0x1E) >> 1; // mask only bits 1-4

    switch (gpio_select)
    {
        case GPIOx_SELECT_OFF:
            ret = GPIOx_SELECT_OFF;
            break;
        case GPIOx_SELECT_MEASURE_READY:
            ret =  GPIOx_SELECT_MEASURE_READY;
            break;
        case GPIOx_SELECT_THRESHOLD_OUTPUT:
            ret =  GPIOx_SELECT_THRESHOLD_OUTPUT;
            break;
        case GPIOx_SELECT_BLANK_IN:
            ret =  GPIOx_SELECT_BLANK_IN;
            break;
        case GPIOx_SELECT_BLANK_OUT:
            ret =  GPIOx_SELECT_BLANK_OUT;
            break;
        case GPIOx_SELECT_START_STOP:
            ret =  GPIOx_SELECT_START_STOP;
            break;
        case GPIOx_SELECT_DISABLED:
            ret =  GPIOx_SELECT_DISABLED;
            break;
        case GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT:
            ret =  GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT;
            break;
        case GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT:
            ret =  GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT;
            break;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_gpio1_polarity(uint8_t device_base_address, uint8_t polarity)
{
    uint8_t gpioModeReg;
    sensor_error ret = SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&polarity);

    gpioModeReg = i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address);

    if (polarity == GPIOx_POLARITY_SELECT_OFF)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg & GPIOx_POLARITY_SELECT_CLEARED),
                               device_base_address);
    }
    else if (polarity == GPIOx_POLARITY_SELECT_ON)
    {
        i2c_write_byte(SYSTEM_MODE_GPIO1,
                               (gpioModeReg | GPIOx_POLARITY_SELECT),
                               device_base_address);
    }
    else
    {
        ret = COMMON_INVALID_PARAMS;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

bool_t common_get_gpio1_polarity(uint8_t device_base_address)
{
    bool_t ret= FALSE;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_MODE_GPIO1, device_base_address) & GPIOx_POLARITY_SELECT)
    {
        ret = TRUE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_history_buffer_enable(uint8_t device_base_address, uint8_t history_buffer_enable)
{
    uint8_t mode;
    sensor_error ret= COMMON_INVALID_PARAMS;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&history_buffer_enable);

    if ((history_buffer_enable == HISTORY_BUFFER_DISABLE) ||
        (history_buffer_enable == HISTORY_BUFFER_ENABLE))
    {
        mode = i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address);
        i2c_write_byte(SYSTEM_HISTORY_CTRL,
                               (mode | history_buffer_enable),
                               device_base_address);
        ret = SENSOR_ERROR_NONE;
    }

    LOG_FUNCTION_END(ret);
		
    return ret;
}

bool_t common_get_history_buffer_enable(uint8_t device_base_address)
{
    bool_t ret = FALSE;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address) & HISTORY_BUFFER_ENABLED)
    {
        ret =  TRUE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_history_buffer_mode(uint8_t device_base_address, uint8_t history_buffer_mode)
{
    uint8_t historyRegVal;
    uint8_t mode=0;
    sensor_error ret= SENSOR_ERROR_NONE;

    LOG_FUNCTION_START((void*)&device_base_address,(void*)&history_buffer_mode);

    historyRegVal = i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address);

    if (history_buffer_mode & HISTORY_BUFFER_ALS_MODE)
    {
        mode = historyRegVal | 0x02; // set bit 1 to enable ALS mode
    }
    else if (history_buffer_mode & HISTORY_BUFFER_RANGING_MODE)
    {
        mode = historyRegVal & 0xFD; // clear bit 1
    }
    else
    {
        ret = COMMON_INVALID_PARAMS;
    }
    i2c_write_byte(SYSTEM_HISTORY_CTRL, mode, device_base_address);

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_history_buffer_mode(uint8_t device_base_address)
{
    int32_t ret = HISTORY_BUFFER_RANGING_MODE;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address) & HISTORY_BUFFER_MODE)
    {
        ret = HISTORY_BUFFER_ALS_MODE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_history_buffer_clear(uint8_t device_base_address)
{
    uint8_t mode;
	
    LOG_FUNCTION_START((void*)&device_base_address);

    mode = i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address);
    mode = mode | 0x04; // set bit 2 to 1 to clear history buffer
    i2c_write_byte(SYSTEM_HISTORY_CTRL, mode, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

bool_t common_get_history_buffer_clear(uint8_t device_base_address)
{
    bool_t ret = FALSE;

    LOG_FUNCTION_START((void*)&device_base_address);
    if (i2c_read_byte(SYSTEM_HISTORY_CTRL, device_base_address) & HISTORY_BUFFER_CLEARED)
    {
        ret= TRUE;
    }
    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error  common_set_system_interrupt_clear_error(uint8_t device_base_address)
{
	
    LOG_FUNCTION_START((void*)&device_base_address);

    i2c_write_byte(SYSTEM_INTERRUPT_CLEAR, INTERRUPT_CLEAR_ERROR, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

uint8_t common_get_error_result_interrupt_status_gpio(uint8_t device_base_address)
{
    uint8_t ret = 0;

    LOG_FUNCTION_START((void*)&device_base_address);

    ret = ( (i2c_read_byte(RESULT_INTERRUPT_STATUS_GPIO, device_base_address) & 0xC0) >> 6 );

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_clear_system_fresh_out_of_reset(uint8_t device_base_address)
{
	
    LOG_FUNCTION_START((void*)&device_base_address);

    i2c_write_byte(SYSTEM_FRESH_OUT_OF_RESET, 0x00, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

bool_t common_get_system_fresh_out_of_reset(uint8_t device_base_address)
{
    uint8_t cFreshOutOfReset = 0x01;
    bool_t ret= FALSE;

    LOG_FUNCTION_START((void*)&device_base_address);

    if (i2c_read_byte(SYSTEM_FRESH_OUT_OF_RESET, device_base_address) & cFreshOutOfReset)
    {
        ret = TRUE;
    }

    LOG_FUNCTION_END(ret);

    return ret;
}

sensor_error common_set_system_group_parameter_hold(uint8_t device_base_address)
{
	
    LOG_FUNCTION_START((void*)&device_base_address);

    i2c_write_byte(SYSTEM_GROUPED_PARAMETER_HOLD, 0x01, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error common_clear_system_group_parameter_hold(uint8_t device_base_address)
{
	
    LOG_FUNCTION_START((void*)&device_base_address);

    i2c_write_byte(SYSTEM_GROUPED_PARAMETER_HOLD, 0x00, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

uint8_t common_get_system_fatal_error_code(uint8_t device_base_address)
{
    uint8_t ret = 0;

    LOG_FUNCTION_START((void*)&device_base_address);

    ret = i2c_read_byte(SYSTEM_FATAL_ERROR_CODE, device_base_address);

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t common_get_system_fatal_error_status(uint8_t device_base_address)
{
    uint8_t ret = 0;

    LOG_FUNCTION_START((void*)&device_base_address);

    ret = i2c_read_byte(SYSTEM_FATAL_ERROR_STATUS, device_base_address);

    LOG_FUNCTION_END(ret);

    return ret;
}

// ??
uint8_t common_get_system_health_check(uint8_t device_base_address)
{
//    return threshold
    return 1; // temp only, will stub is here!
}

