#ifndef SSD1327_H
#define SSD1327_H

#include "mbed.h"

/** 96x96 pixel SSD1327 SeeedStudio 1.12" OLED display

based on https://github.com/blitz/mbed-arch-pro

Example:
@code
#include "mbed.h"
#include "SSD1327.h"
#include "defaultfont.h"

//Pin Defines for I2C Bus
#define D_SDA                  D3
#define D_SCL                  D6
I2C i2c(D_SDA, D_SCL);

// Instantiate OLED
static SSD1327 oled { i2c };
 

 int main() {
         oled.set_font(default_font,8,8);
         oled.set_text_rc(0,0);
         oled.puts("Hello world!");
         while (1) {wait(0.1);}
}
@endcode
*/

class SSD1327 {
    I2C &i2c;


public:
    /* Create SSD1327 object
    */
    SSD1327(I2C &i2c);

    const static uint8_t address   = 0x3c << 1;
    const static uint8_t cmd_mode  = 0x80;
    const static uint8_t data_mode = 0x40;

    const static uint8_t PIXELS         = 96;
    const static uint8_t PIXEL_PER_CHAR = 8; // if default font used

    const static uint8_t ROWS = PIXELS / PIXEL_PER_CHAR;
    const static uint8_t COLS = PIXELS / PIXEL_PER_CHAR;
    
    /** Set Gray Level 
    * @param level - new gray level  (0..0xF)
    */
    void set_gray_level(unsigned char level);
    
    /** Set text font
    * @param Font - font array
    * @param width - font width (only 8, 16, 24 or 32)
    * @param height - font height
    */
    void set_font(const uint8_t *Font, int width, int height);
    
    /** Set new Row and Column
    * @param row - new row (0..95)
    * @param column - new column (0..95)
    */
    void set_rc(unsigned char row, unsigned char column);
    
    /** Set new text Row and Column
    * Used after set_font
    * @param row - new row (0..96/font_height-1)
    * @param column - new column (0..96/font_width-1)
    */
    void set_text_rc(unsigned char row, unsigned char column);
    
    /** Put char into display using current font
    * @param C - printed char
    */
    void putc(unsigned char C);

    /** Put string into display using current font
    * @param str - printed string
    */
    void puts(const char *str);
    
    /** Clear display
    */
    void clear();
    
    // Service low-level functions

    bool send(char mode, const char data[], size_t len);

    void tribyte_cmd(uint8_t a, uint8_t b, uint8_t c);
    
    void set_column_address(uint8_t start, uint8_t end);

    void set_row_address(uint8_t start, uint8_t end);
    
    //bool sendf(char mode, char data[], size_t len);
    
    bool sendx(char mode, char data[], size_t len);
    
    
};

#endif // SSD1327_H

// EOF
