/** SongPlayer class.
 *  Used for to play MIDI notes on Speaker/Buzzer using PWM and timer interrupts (single channel).
 *  based on https://developer.mbed.org/users/4180_1/code/song_demo_PWM/
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "SongPlayer.h"
 * #include "sound.h" // test sound
 * DigitalOut led1(LED1);
 * 
 * int main()
 * {
 * // setup instance of new SongPlayer class, mySpeaker using pin D9
 * // the pin must be a PWM output pin
 *    SongPlayer mySpeaker(D9);
 * // Start song and return once playing starts
 *    mySpeaker.PlaySong(score3);
 *    // loops forever while song continues to play to end using interrupts
 *    int cnt=0;
 *    while(1) {
 *        led1 = !led1;
 *        wait(.1);
 *        cnt++;
 *        if (cnt>100) mySpeaker.StopPlay(); // play 10 second, then stop
 *    }
 * }
 * @endcode
 */ 
#ifndef __SONGPLAYER_H
#define __SONGPLAYER_H
 
 
#include "mbed.h"
#include "miditable.h"

#define CMD_PLAYNOTE    0x90    /* play a note: low nibble is generator #, note is next byte */
#define CMD_STOPNOTE    0x80    /* stop a note: low nibble is generator # */
#define CMD_RESTART 0xe0    /* restart the score from the beginning */
#define CMD_STOP    0xf0    /* stop playing */
/* if CMD < 0x80, then the other 7 bits and the next byte are a 15-bit big-endian number of msec to wait */


class SongPlayer
{
public:
    /** Create SongPlayer instance
     * @param pin PWM output pin
         */
    SongPlayer(PinName pin) : _pin(pin) {
// _pin(pin) means pass pin to the constructor
    }
   /** Play the MIDI notes sequence
         *
         * @param notes MIDI notes sequence array
         */
    void PlaySong(const uint8_t notes[], float volume=1.0) {
        vol = volume;
        notecount = 0;
        currentnote=0; currentduration =0.001;
        _pin.period(1.0/currentnote);
        _pin = volume/2.0;
        isrun=true;
        noteduration.attach(this,&SongPlayer::nextnote, currentduration);
        // setup timer to interrupt for next note to play
        notesptr = notes;
        //returns after first note starts to play
    }
   /** Stop play notes
         */
    void StopPlay(){
        isrun=false;
    }
    void nextnote();
    /** return true if song is playing now
        */
    bool playing() {return isrun;}
private:
    Timeout noteduration;
    float currentnote, currentduration;
    PwmOut _pin;
    bool isrun;
    int notecount;
    float vol;
    const uint8_t * notesptr;
};
//Interrupt Routine to play next note
void SongPlayer::nextnote()
{
    uint8_t cmd,cmdx;
    _pin = 0.0;
    if (!isrun) return;
    while (1) {
        cmd = notesptr[notecount++];
        cmdx=cmd & 0xf0;
        if (cmd < 0x80) { /* wait count in msec. */
            currentduration = (((uint16_t)cmd << 8) | notesptr[notecount++])/1000.0;
            _pin.period(1.0/currentnote);
            noteduration.attach(this,&SongPlayer::nextnote, currentduration);
            _pin = vol/2.0;
            break;
        }
        if (cmdx == CMD_STOPNOTE) { /* stop note */
            currentnote=0;
        }
        else if (cmdx == CMD_PLAYNOTE) { /* play note */
            currentnote=midi[notesptr[notecount++]];
        }
        else if (cmdx == CMD_RESTART) { /* restart score */
//            notecount=0;
        }
        else if (cmdx == CMD_STOP) { /* stop score */
            isrun=false;
            _pin = 0.0; //turn off on last note 
            break;
        }
     }   
}

#endif