/** Two axis analog joystick driver
 *
 * Example:
 * @code
 * // Print joystick values
 *
 * #include "mbed.h"
 * #include "Joystick.h"
 *
 * Ticker joysticTicker;
 * Joystick joystick(PA_1, PA_0); //Two analog in pins
 * volatile bool processJoystick = false;
 *
 * void isrJoystick()
 * {
 *     processJoystick = true;
 * }
 *
 * int main()
 * {
 *     joystick.setFlipX(true);
 *     joystick.setRange(100);
 *     while (!joystick.isCalibrated())
 *     {
 *         joystick.process();
 *         wait(0.05);
 *     }
 *     joysticTicker.attach(&isrJoystick, 0.1);
 *     while (true)
 *     {
 *         if (processJoystick)
 *         {
 *             processJoystick = false;
 *             joystick.process();
 *             printf("X=%d Y=%d\r\n", joystick.getX(), joystick.getY());
 *         }
 *     }
 * }
 *
 */

#ifndef Joystick_h
#define Joystick_h

#include "mbed.h"

#ifndef JOYSTICK_CALIBRATION_CYCLES
#define JOYSTICK_CALIBRATION_CYCLES 25
#endif
#ifndef JOYSTICK_ADC_MAX
#define JOYSTICK_ADC_MAX 0xFFFF
#endif
#define JOYSTICK_CENTER_DEADZONE JOYSTICK_ADC_MAX * _centerDeadzone

struct JoystickValue
{
    JoystickValue()
    : x(0)
    , y(0)
    {};
    JoystickValue(int a)
    : x(a)
    , y(a)
    {};
    JoystickValue(int ax, int ay)
    : x(ax)
    , y(ay)
    {};
    int x;
    int y;
};

class Joystick
{
public:
    Joystick(PinName pinX, PinName pinY, float processInterval = 0, float centerDeadzone = 0.03f, void(*onChange)(JoystickValue value, JoystickValue prevValue) = NULL)
    : _xIn(pinX)
    , _yIn(pinY)
    , _center(JOYSTICK_ADC_MAX / 2)
    , _joyValue(0)
    , _calibrated(false)
    , _calibrationCounter(0)
    , _range(JOYSTICK_ADC_MAX / 2)
    , _centerDeadzone(centerDeadzone)
    , _swapXY(false)
    , _flipX(false)
    , _flipY(false)
    , _delta(0)
    , _locked(false)
    , _onChange(onChange)
    {
        setAutoProcessInterval(processInterval);
    };

    void process(void);

    void setAutoProcessInterval(float processInterval)
    {
        _ticker.detach();
        if (processInterval != 0) _ticker.attach(Callback<void()>(this, &Joystick::process), processInterval);
    };

    void calibrate()
    {
        _center.x = JOYSTICK_ADC_MAX / 2;
        _center.y = JOYSTICK_ADC_MAX / 2;
        _calibrationCounter = 0;
        _calibrated = false;
    };

    inline float getCenterDeadzone() const
    {
        return _centerDeadzone;
    };

    inline void setCenterDeadzone(float centerDeadzone)
    {
        if (centerDeadzone < 0 || centerDeadzone > 1) return;
        _centerDeadzone = centerDeadzone;
    };

    inline bool isCalibrated() const
    {
        return _calibrated;
    };

    inline bool isLocked() const
    {
        return _locked;
    };

    inline void setLocked(bool locked)
    {
        _locked = locked;
    };

    inline bool isSwapXY() const
    {
        return _swapXY;
    };

    inline void setSwapXY(bool swapXY)
    {
        _swapXY = swapXY;
    };

    inline bool isFlipX() const
    {
        return _flipX;
    };

    inline void setFlipX(bool flipX)
    {
        _flipX = flipX;
    };

    inline bool isFlipY() const
    {
        return _flipY;
    };

    inline void setFlipY(bool flipY)
    {
        _flipY = flipY;
    };

    inline int getRange() const
    {
        return _range;
    };

    inline void setRange(int range)
    {
        _range = range;
    };

    inline int getDelta() const
    {
        return _delta;
    };

    inline void setDelta(int delta)
    {
        _delta = delta;
    };

    inline int getX() const
    {
        return _joyValue.x;
    };

    inline int getY() const
    {
        return _joyValue.y;
    };

    inline JoystickValue getCurrentValue() const
    {
        return _joyValue;
    };

private:
    AnalogIn _xIn;
    AnalogIn _yIn;
    JoystickValue _center;
    JoystickValue _joyValue;
    bool _calibrated;
    uint16_t _calibrationCounter;
    int _range;
    float _centerDeadzone;
    bool _swapXY;
    bool _flipX;
    bool _flipY;
    int _delta;
    bool _locked;
    void(*_onChange)(JoystickValue value, JoystickValue prevValue);
    Ticker _ticker;
};

#endif //Joystick_h

