/**
* @file    eeram_main.cpp
* @brief   Example program for Microchip I2C EERAM devices (47x04 and 47x16)
* @author  Mark Peter Vargha, vmp@varghamarkpeter.hu
* @version 1.0.0
*
* Copyright (c) 2017
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#include "mbed.h"
#include "EERAM.h"

#define PIN_I2C_SDA PC_9
#define PIN_I2C_SCL PA_8

//#define I2C_FREQUENCY 1000000
#define I2C_FREQUENCY 400000

struct StorageContainer
{
    float f;
    int i;
    short s[4];
};

Serial serial(PA_9, PA_10);  //Tx, Rx
I2C i2c(PIN_I2C_SDA, PIN_I2C_SCL); //SDA, SCL
EERAM eeram(i2c, 2048);

void printI2C()
{
    //0x41  Discovery Touch
    //0x18  EERAM control
    //0x50  EERAM memory

    int error;
    int address;
    int nDevices = 0;

    serial.printf("Scanning I2C devices...\r\n");

    for(address = 1; address < 127; address++ )
    {
        i2c.start();
        error = i2c.write(address << 1); //We shift it left because mbed takes in 8 bit addreses
        i2c.stop();
        if (error == 1)
        {
            serial.printf("I2C device found at address 0x%X\r\n", address); //Returns 7-bit addres
            nDevices++;
        }

    }
    serial.printf("I2C scan finished.\r\n");
    if (nDevices == 0)
    {
        serial.printf("No I2C devices found.\r\n");
    }

}

void fillTestData(char data[], uint8_t start, int length)
{
    for (int i = 0; i < length; i++) data[i] = start + i;
}

void eeramDataTest()
{
    const int testDataLength = 16;
    char data[testDataLength];

    //Write
    eeram.fillMemory(0xFF);

    fillTestData(data, 0x0, testDataLength);
    serial.printf("Write %d bytes to 0x0: %d\r\n", testDataLength, eeram.writeBytes(0x0, data, testDataLength));

    fillTestData(data, 0x50, testDataLength);
    serial.printf("Write %d bytes to 0x500: %d\r\n", testDataLength, eeram.writeBytes(0x500, data, testDataLength));

    fillTestData(data, 0x70, testDataLength);
    serial.printf("Write %d bytes to 0x700: %d\r\n", testDataLength, eeram.writeBytes(0x700, data, testDataLength));

    //Dump
    serial.printf("Dump contents 0x0, 16\r\n");
    eeram.dump(serial, 0x0, testDataLength);
    serial.printf("Dump contents 0x500, 16\r\n");
    eeram.dump(serial, 0x500, testDataLength);
    serial.printf("Dump contents 0x700, 16\r\n");
    eeram.dump(serial, 0x700, testDataLength);
    //serial.printf("Dump all\r\n");
    //eeram.dump(serial);
    serial.printf("Dump done\r\n");

    //Read back
    fillTestData(data, 0x0, testDataLength);
    serial.printf("Read back 16 bytes from 0x500: %d\r\n", eeram.readBytes(0x500, data, testDataLength));
    serial.printf("%.4X ", 0x500);
    for (int i = 0; i < testDataLength; i++)
    {
        serial.printf("%.2X ", data[i]);
    }
    serial.printf("\r\n");

    //Read and write individual values
    const uint16_t START_ADDRESS = 0x400;
    uint16_t addressPtr = START_ADDRESS;
    int intToWrite = -76324;
    unsigned short ushortToWrite = 4395;
    float floatToWrite = 0.1976f;
    serial.printf("# # #\r\n");
    serial.printf("Write individual values, start address = 0x%X\r\n", START_ADDRESS);
    serial.printf("Int to write: %d\r\n", intToWrite);
    serial.printf("Unsigned short to write: %d\r\n", ushortToWrite);
    serial.printf("Float to write: %f\r\n", floatToWrite);
    addressPtr += eeram.write(addressPtr, &intToWrite);
    addressPtr += eeram.write(addressPtr, &ushortToWrite);
    addressPtr += eeram.write(addressPtr, &floatToWrite);

    addressPtr = START_ADDRESS;
    int intToRead = 0;
    unsigned short ushortToRead = 0;
    float floatToRead = 0.0;
    addressPtr += eeram.read(addressPtr, &intToRead);
    addressPtr += eeram.read(addressPtr, &ushortToRead);
    addressPtr += eeram.read(addressPtr, &floatToRead);
    serial.printf("Read back individual values\r\n");
    serial.printf("Int to read: %d\r\n", intToRead);
    serial.printf("Unsigned short to read: %d\r\n", ushortToRead);
    serial.printf("Float to read: %f\r\n", floatToRead);
    int length = addressPtr - START_ADDRESS + 6;
    serial.printf("Dump %d bytes from 0x%X:\r\n", length, START_ADDRESS);
    eeram.dump(serial, START_ADDRESS, length);

    //Read and write array
    serial.printf("# # #\r\nTest arrays\r\nWrite:\r\n");
    const int arraySize = 4;
    const uint16_t address = 0x300;
    float floatArrayToWrite[arraySize] = {1976.09f, 1979.04f, 2013.04f, 2015.11f};
    for (int i = 0; i < arraySize; i++) serial.printf("%f ", floatArrayToWrite[i]);
    serial.printf("\r\nRead:\r\n");
    eeram.write(address, floatArrayToWrite, arraySize);
    float floatArrayToRead[arraySize];
    eeram.read(address, floatArrayToRead, arraySize);
    for (int i = 0; i < arraySize; i++) serial.printf("%f ", floatArrayToRead[i]);
    serial.printf("\r\n");

    //Read and write custom types
    serial.printf("# # #\r\nTest custom types\r\n");
    const int customTypeStart = 0x200;
    StorageContainer containerToWrite;
    containerToWrite.f = 1976.09;
    containerToWrite.i = 2015;
    containerToWrite.s[0] = -1;
    containerToWrite.s[1] = -20;
    containerToWrite.s[2] = -30;
    containerToWrite.s[3] = -40;
    serial.printf("Write container's float: %f\r\n", containerToWrite.f);
    serial.printf("Write container's int: %d\r\n", containerToWrite.i);
    serial.printf("Write container's short array: ");
    for (int i = 0; i < 4; i++) serial.printf("%d ", containerToWrite.s[i]);
    serial.printf("\r\n");
    eeram.write(customTypeStart, &containerToWrite);
    StorageContainer containerToRead;
    eeram.read(customTypeStart, &containerToRead);
    serial.printf("Read container's float: %f\r\n", containerToRead.f);
    serial.printf("Read container's int: %d\r\n", containerToRead.i);
    serial.printf("Read container's short array: ");
    for (int i = 0; i < 4; i++) serial.printf("%d ", containerToRead.s[i]);
    serial.printf("\r\n");
}

void eeramRegisterTest()
{
    serial.printf("# # #\r\n");
    eeram.dumpRegisters(serial);
}

int main()
{
    serial.baud(460800);
    i2c.frequency(I2C_FREQUENCY); //Hz
    serial.printf("\r\nI2C EERAM example\r\n");
    serial.printf("MBED version: %d.%d.%d\r\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

    printI2C();

    serial.printf("Is EERAM device ready?\r\n");
    while (!eeram.isReady());
    serial.printf("Device is ready.\r\n");

    eeram.readStatus();
    eeram.setAutoStoreEnabled(true);
    eeram.setProtectedMemoryArea(U64);
    eeram.writeStatusIfChanged(true);
    serial.printf("Status: %.2X\r\n", eeram.getStatus());

    eeramDataTest();

    eeramRegisterTest();

    //eeram.store(true);
    //eeram.recall(true);

    while (true)
    {

    }
}

/*
#include "mbed.h"
#include "EERAM.h"

EERAM eeram(PC_9, PA_8, 2048); //SDA, SCL

int main()
{
    if (!eeram.isReady(100)) //Checks device with 100 ms timeout
    {
        printf("Device is not present.");
        while (1);
    }
    eeram.readStatus(); //Reads status register
    eeram.setAutoStoreEnabled(true, true); //Set auto store on power down to true and stores if not stored before
    while (1)
    {
        char dataStore[16] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16};
        eeram.writeBytes(0x100, dataStore, 16); //We can not wear EEPROM out, so it is ok to write data to the device frequently.
        wait(2.0);
        char dataRead[16];
        eeram.readBytes(0x100, dataRead, 16);
        wait(2.0);
        float floatToWrite = -1.976f;
        const uint16_t address = 0x200;
        eeram.write(address, &floatToWrite);
        wait(2.0);
        float floatToRead = 0;
        eeram.read(address, &floatToWrite);
        wait(2.0);
    }
}

*/