/**
* @file    Button.h
* @brief   Button library
* @author  Mark Peter Vargha, vmp@varghamarkpeter.hu
*
* Copyright (c) 2017
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#ifndef BUTTON_H_INCLUDED
#define BUTTON_H_INCLUDED

#include "mbed.h"

enum ButtonEvent
{
    BUTTON_UNKNOWN = 0, BUTTON_DOWN, BUTTON_UP, BUTTON_PRESS, BUTTON_LONG_PRESS
};

class Button
{
public:
    Button(PinName pinName, PinMode pinMode, int activeState, Callback<void(ButtonEvent, PinName)> onButtonEvent = 0, uint16_t autoRepeatTime = 0, uint16_t longPressTime = 600, uint16_t shortPressTime = 30)
    : _pinName(pinName)
    , _in(pinName)
    , _shortPressTime(shortPressTime)
    , _longPressTime(longPressTime)
    , _autoRepeatTime(autoRepeatTime)
    , _activeState(activeState)
    , _prevState(0)
    , _fireUpDown(false)
    , _onButtonEvent(onButtonEvent)
    {
        _in.mode(pinMode);
        startInterrupt();
    };
    void startInterrupt();
    void stopInterrupt();
    inline void setAutoRepeatTime_ms(uint16_t autoRepeatTime)
    {
        _autoRepeatTime = autoRepeatTime;
    };
    inline uint16_t getAutoRepeatTime_ms() const
    {
        return _autoRepeatTime;
    }
    inline int read()
    {
        return _in.read() == _activeState;
    };
    operator int()
    {
        return read();
    };
    void setFireUpDown(bool fire)
    {
        _fireUpDown = fire;
    };
    inline bool isFireUpDown() const
    {
        return _fireUpDown;
    };
    void setShortPressTime_ms(uint16_t time_ms)
    {
        _shortPressTime = time_ms;
    };
    inline uint16_t getShortPressTime_ms() const
    {
        return _shortPressTime;
    };
    void setLongPressTime_ms(uint16_t time_ms)
    {
        _longPressTime = time_ms;
    };
    inline uint16_t getLongPressTime_ms() const
    {
        return _longPressTime;
    };
    void attach(Callback<void(ButtonEvent, PinName)> onButtonEvent)
    {
        _onButtonEvent = onButtonEvent;
    };
    void detach()
    {
        _onButtonEvent = 0;
    };
private:
    PinName _pinName;
    InterruptIn _in;
    Timer _timer;
    Ticker _tickerAutoRepeat;
    Timeout _autoRepeatStarter;
    uint16_t _shortPressTime;
    uint16_t _longPressTime;
    uint16_t _autoRepeatTime;
    int _activeState;
    int _prevState;
    bool _fireUpDown;
    Callback<void(ButtonEvent, PinName)> _onButtonEvent;
    void isrFall(void);
    void isrRise(void);
    void onPress(void);
    void onRelease(void);
    void autoRepeat(void);
    void startAutoRepeat(void);
};

#endif /* BUTTON_H_INCLUDED */
