//=============================================================================
//  @author vaifreak
//  @brief  Guitar Effector for "mbed application board"
//=============================================================================
#include "mbed.h"
#include <new>
#include <list>

#include "MyInputUtil.h"
#include "Menu.h"
#include "LevelMeter.h"

#include "Filter.h"
#include "Drive.h"
#include "Delay.h"


//---------------------------------------------
//---------------------------------------------
AnalogOut a_out(p18);
AnalogIn a_in(p17);

Filter *low_pass;
Filter *high_pass;
Drive *drive;
Delay *delay;

list<EffectUnitBase*> g_EffectList;

Menu menu;
LevelMeter level_meter;

Ticker menu_ticker;
Ticker process_ticker;

float g_sample_rate = 22050.0f;


//---------------------------------------------
// Forward declaration
//---------------------------------------------
void process();
void update_display();


//---------------------------------------------
extern "C"
void HardFault_Handler() {
    printf("Hard Fault!\r\n");
    while(1);
}

//---------------------------------------------
void no_memory () {
    printf("panic: can't allocate to memory!\r\n");
    exit(-1);
}

//---------------------------------------------
//
//---------------------------------------------
int main()
{
    printf("mbed effector, start!\n");
    set_new_handler(no_memory); // new handler function

    // create effect unit     
    low_pass = new Filter( Filter::LPF );
    high_pass = new Filter( Filter::HPF );
    drive = new Drive();
    delay = new Delay();

    // add to effect processing list
    g_EffectList.push_back(low_pass);
    g_EffectList.push_back(high_pass);
    g_EffectList.push_back(drive);
    g_EffectList.push_back(delay);

    menu.Init( g_sample_rate, low_pass, high_pass, drive, delay );

    process_ticker.attach( &process, 1.0/g_sample_rate );
    menu_ticker.attach( &update_display, 0.1 );
    
    //---- main loop ----
    //while(1)
    //{
    //}
}

//---------------------------------------------
//
//---------------------------------------------
void process()
{
    float val = (a_in * 2.0f) - 1.0f; //convert [0.0 ~ 1.0] to [-1.0 ~ 1.0]
    val = val * menu.inputGain;
    level_meter.Update( fabsf(val) ); // LED level meter.(for Input)

    //---- effect process ----
    list<EffectUnitBase*>::iterator it = g_EffectList.begin();
    while( it != g_EffectList.end() )
    {
        val = (*it)->Process( val );
        ++it;
    }

    val = val * menu.masterVolume;
    //level_meter.Update( fabsf(val) ); // LED level meter.(for Output)
    val = (val + 1.0f) * 0.5f; //convert [-1.0 ~ 1.0] to [0.0 ~ 1.0]
    
    //clipping. just to be safe. 
    if(val > 1.0f) val = 1.0f;
    else if(val < 0.0f) val =0.0f;

    a_out = val;
}

//---------------------------------------------
//
//---------------------------------------------
void update_display()
{
    menu.Update();
}


