//=============================================================================
//  @author vaifreak
//  @brief  menu display & effect parameter Control.
//=============================================================================
#pragma once

#include "Filter.h"
#include "Drive.h"
#include "Delay.h"
#include "MyInputUtil.h"

#include "C12832.h"

#define EFFECT_ON_INIT true

//---------------------------------------------
// 
//---------------------------------------------
class Menu
{
public:
    float inputGain;
    float masterVolume;

private:
    //function type.
    enum {
        FILTER_LPF,
        FILTER_HPF,
        DRIVE,
        DELAY,
        MASTER,
        
        FUNCTION_MAX
    };

    //param for LPF
    float filter_lpf_freq;
    float filter_lpf_Q;

    //param for HPF
    float filter_hpf_freq;
    float filter_hpf_Q;

    //param for Drive
    float drive_gain;
    float drive_volume;

    //param for delay
    float delay_time;
    float delay_feedback;
    float delay_mix;

    float sample_rate;

    bool isOn[FUNCTION_MAX];

    bool dirtyFlag;
    void SetDirty(){ dirtyFlag = true; }
    bool IsDirty(){ return dirtyFlag; }
    void ClearDirty(){ dirtyFlag = false; }

    C12832 *lcd;
    MyVolume *pot1;
    MyVolume *pot2;
    MyButton *stick_U;
    MyButton *stick_D;
    MyButton *stick_L;
    MyButton *stick_R;
    MyButton *stick_C;

    Filter *filter_lpf;
    Filter *filter_hpf;
    Drive *drive;
    Delay *delay;

    int current_function;
   
public:
    //-----------------------------------------
    //
    //-----------------------------------------
    Menu()
    {
        lcd = new C12832(p5, p7, p6, p8, p11);
        pot1 = new MyVolume(p19);
        pot2 = new MyVolume(p20);
        stick_U = new MyButton(p15);
        stick_D = new MyButton(p12);
        stick_L = new MyButton(p13);
        stick_R = new MyButton(p16);
        stick_C = new MyButton(p14);
        
        printf("Menu constructed.\n");
    }

    //-----------------------------------------
    //
    //-----------------------------------------
    void Init(
        float sample_rate_,
        Filter *filter_lpf_,
        Filter *filter_hpf_,
        Drive *drive_,
        Delay *delay_ )
    {
        sample_rate = sample_rate_;
        
        filter_lpf = filter_lpf_;
        filter_hpf = filter_hpf_;
        drive = drive_;
        delay = delay_;

        for(int i=0; i<FUNCTION_MAX; i++){
            isOn[i] = EFFECT_ON_INIT;
        }

        current_function = 0;

        filter_lpf_freq = filter_lpf->GetFreq();
        filter_lpf_Q = filter_lpf->GetQ();
        UpdateLpfParam();

        filter_hpf_freq = filter_hpf->GetFreq();
        filter_hpf_Q = filter_hpf->GetQ();
        UpdateHpfParam();

        drive_gain = 9.0f;
        drive_volume = 0.45f;
        UpdateDriveParam();
        
        delay_time = 0.8f;
        delay_feedback = 0.5f;
        delay_mix = 0.5f;
        UpdateDelayParam();

        inputGain = 1.0f;
        masterVolume = 1.0f;

        UpdateDisplay();
    }
        
    //-----------------------------------------
    //
    //-----------------------------------------
    void Update()
    {
        pot1->Update();
        pot2->Update();
        stick_U->Update();
        stick_D->Update();
        stick_L->Update();
        stick_R->Update();
        stick_C->Update();
        
        //---- select function 
        if( stick_U->trig_on ){
            current_function --;
            if(current_function < 0) current_function = FUNCTION_MAX - 1;
            SetDirty();
        }
        if( stick_D->trig_on ){
            current_function ++;
            if(current_function >= FUNCTION_MAX) current_function = 0;
            SetDirty();
        }

        //---- effect ON/OFF 
        if( stick_C->trig_on ){
            isOn[ current_function ] = !isOn[ current_function ];
            SetDirty();
        }

        UpdateParameter();

        // update display.        
        if( IsDirty() )
        {
            UpdateDisplay();
            ClearDirty();
        }
    }

private:
    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateDisplay()
    {
        lcd->cls();
        lcd->locate(0,0);

        bool is_on = isOn[current_function];
        
        switch( current_function )
        {
        case FILTER_LPF:
            lcd->printf("LowPass : %s", (is_on?"ON ":"off") );
            lcd->locate(0,10);
            lcd->printf("Freq.:%.0f Q:%.2f ", filter_lpf_freq, filter_lpf_Q );
            break;
            
        case FILTER_HPF:
            lcd->printf("HighPass: %s", (is_on?"ON ":"off") );
            lcd->locate(0,10);
            lcd->printf("Freq.:%.0f Q:%.2f ", filter_hpf_freq, filter_hpf_Q );
            break;

        case DRIVE:
            lcd->printf("Drive   : %s", (is_on?"ON ":"off") );
            lcd->locate(0,10);
            lcd->printf("Gain:%.2f Vol.:%.2f ", drive_gain, drive_volume );
            break;
                        
        case DELAY:
            lcd->printf("Delay   : %s", (is_on?"ON ":"off") );
            lcd->locate(0,10);
            lcd->printf("Time:%4dmsec Mix:%.2f ", delay->GetDelayTimeInMSec(sample_rate), delay_mix);
            break;
            
        case MASTER:
            lcd->printf("InputGain : %.2f", inputGain );
            lcd->locate(0,10);
            lcd->printf("MasterVol: %.2f", masterVolume );
            break;
        }

    }
    
    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateParameter()
    {
        switch( current_function )
        {
        case FILTER_LPF:
            {
                if( pot1->dirtyFlag ) {
                    filter_lpf_freq = RateToParam( 2000.0f, 6000.0f, pot1->val );
                    SetDirty();
                }
                if( pot2->dirtyFlag ) {
                    filter_lpf_Q = RateToParam( 0.3f, 2.0f, pot2->val );
                    SetDirty();
                }
                if( IsDirty() ){
                    UpdateLpfParam();
                }
            }
            break;
 
         case FILTER_HPF:
            {
                if( pot1->dirtyFlag ) {
                    filter_hpf_freq = RateToParam( 50.0f, 700.0f, pot1->val );
                    SetDirty();
                }
                if( pot2->dirtyFlag ) {
                    filter_hpf_Q = RateToParam( 0.3f, 2.0f, pot2->val );
                    SetDirty();
                }
                if( IsDirty() ){
                    UpdateHpfParam();
                }
            }
            break;
           
        case DRIVE:
            {
                if( pot1->dirtyFlag ) {
                    drive_gain = pot1->val * 20.0f;
                    SetDirty();
                }
                if( pot2->dirtyFlag ) {
                    drive_volume = pot2->val * 2.0f;
                    SetDirty();
                }
                if( IsDirty() ){
                    UpdateDriveParam();
                }
            }
            break;
            
            case DELAY:
            {
                if( pot1->dirtyFlag ) {
                    delay_time = pot1->val;
                    SetDirty();
                }
                if( pot2->dirtyFlag ) {
                    delay_mix = pot2->val;
                    SetDirty();
                }
                if( IsDirty() ){
                    UpdateDelayParam();
                }
            }
            break;
            
        case MASTER:
            if( pot1->dirtyFlag )
            {
                inputGain = pot1->val * 4.0f;
                SetDirty();
            }
            if( pot2->dirtyFlag )
            {
                masterVolume = pot2->val * 2.0f;
                SetDirty();
            }
            break;
        }
    }

    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateLpfParam()
    {
        filter_lpf->isBypass = !isOn[FILTER_LPF];
        filter_lpf->set_LPF( sample_rate, filter_lpf_freq, filter_lpf_Q );
    }

    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateHpfParam()
    {
        filter_hpf->isBypass = !isOn[FILTER_HPF];
        filter_hpf->set_HPF( sample_rate, filter_hpf_freq, filter_hpf_Q );
    }

    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateDriveParam()
    {
        drive->isBypass = !isOn[DRIVE];
        drive->gain = drive_gain;
        drive->vol  = drive_volume;
    }
    
    //-----------------------------------------
    //
    //-----------------------------------------
    void UpdateDelayParam()
    {
        delay->isBypass = !isOn[DELAY];
        delay->delay_time = delay_time;
        delay->feedback = delay_feedback;
        delay->effect_level = delay_mix;
    }
    
    //-----------------------------------------
    //
    //-----------------------------------------
    float RateToParam( float min, float max, float rate )
    {
        return (max - min) * rate + min;
    }
};

