//=============================================================================
//  @author vaifreak
//  @brief  filter unit (low pass / high pass)
//=============================================================================
#pragma once

#include "EffectUnitBase.h"

//---------------------------------------------
// 
//---------------------------------------------
class Filter : public EffectUnitBase
{
public:
    enum {
        LPF,
        HPF
    };

private:
    int type;
    float freq;
    float Q;

    float in1, in2;
    float out1, out2;
    float a0, a1, a2, b0, b1, b2;
    static const float PI = (3.14159265f);
    
public:
    int GetType(){ return type; }
    float GetFreq(){ return freq; }
    float GetQ(){ return Q; }

    Filter( int type_ )
    {
        printf("Filter\n");
        
        in1 = 0.0f;
        in2 = 0.0f;
        out1 = 0.0f;
        out2 = 0.0f;
        
        type = type_;
        if(type == LPF)
            set_LPF();
        else 
            set_HPF();
    }

    // LPF (Low Pass Filter)    
    void set_LPF( float samplerate = 22050.0f, float f = 4500.0f, float q = 1.0f )
    {
        printf("set_LPF\n");

        this->freq = f;
        this->Q = q;
        
        float omega = 2.0f * PI * f / samplerate;
        float alpha = sinf(omega) / (2.0f * q);
         
        a0 =  1.0f + alpha;
        a1 = -2.0f * cosf(omega);
        a2 =  1.0f - alpha;
        b0 = (1.0f - cosf(omega)) / 2.0f;
        b1 =  1.0f - cosf(omega);
        b2 = (1.0f - cosf(omega)) / 2.0f;
    }

    // HPF (High Pass Filter)    
    void set_HPF( float samplerate = 22050.0f, float f = 250.0f, float q = 1.0f )
    {
        printf("set_HPF\n");

        this->freq = f;
        this->Q = q;

        float omega = 2.0f * PI * f / samplerate;
        float alpha = sinf(omega) / (2.0f * q);
 
        a0 =   1.0f + alpha;
        a1 =  -2.0f * cosf(omega);
        a2 =   1.0f - alpha;
        b0 =  (1.0f + cosf(omega)) / 2.0f;
        b1 = -(1.0f + cosf(omega));
        b2 =  (1.0f + cosf(omega)) / 2.0f;
    }

protected:
    virtual float ProcessConcrete(float input)
    {
        float output = b0/a0 * input + b1/a0 * in1  + b2/a0 * in2
                                     - a1/a0 * out1 - a2/a0 * out2;
        in2  = in1;
        in1  = input;
        out2 = out1; 
        out1 = output;
        
        return output;
    }
};