/* ---------------------------------------------------------------------------------
Copyright 2015 Marijn Billiet

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   
 --------------------------------------------------------------------------------- */

#ifndef MBED_AD5933_H
#include "mbed.h"
#define MBED_AD5933_H


/** AD5933 class.
 *  Library to communicate with the AD5933 impedance meter chip.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ad5933.h"
 * #define WAIT_TIME_MS 3000
 * #define START_FREQ 10000
 * #define STEP_FREQ 100
 * 
 * AD5933 Zmeter(P3_4, P3_5, false);
 * 
 * int main()
 * {
 *     
 *     printf("This is AD5933 test running on Mbed OS %d.%d.%d.\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
 *     float temp = Zmeter.getTemperature();
 *     Zmeter.initFrequencySweepParam(START_FREQ, STEP_FREQ, 10, 5, true, 1);
 * 
 *     int i=0;
 *     printf("Sweep started \n");
 *     int status = Zmeter.Measure(false);
 *     while (status != 7)
 *     {
 *         temp = Zmeter.getTemperature();
 *         printf("freq = %d Hz, real = %d, img = %d, T = %.2fC\n", START_FREQ + STEP_FREQ*i, Zmeter.real, Zmeter.imaginary, temp); 
 *         thread_sleep_for(WAIT_TIME_MS);
 *         status = Zmeter.Measure(true);
 *         i++;
 *     }
 * 
 *     printf("Sweep completed \n");
}
 * @endcode
 */
class AD5933
{
public:
    /** Create AD5933 instance
    @param sda I2C data line pin
    @param scl I2C clock line pin
    @param extClk source of the Clock signal:  true = external;  false = internal
    */
    AD5933(PinName sda, PinName scl, bool extClk);

    /** Time to wait before the ADC starts to sample
    @param nrOfCycles Number of cycles (outputfrequency) to wait for sampling can start  (max 2044)
    @return true on succes, false on failure
    */
    bool setSettlingTime(unsigned int nrOfCycles);

    /** Set the gain of the pre-amplification and the output voltage range
    @param PGA Gain of the pre-amplifier (true = x1; false = x5)
    @param RangeNr Set the output voltage (1 = 2V; 2 = 1V; 3 = 400mV; 4 = 200mV)
    @return true on succes, false on failure
    */
    bool setAnalogCircuit(bool PGA, int RangeNr);

    /** Measure Impedance
    @param increment (true = goto next frequency in sweep; false = remeasure at current frequency)
    @return true on succes, false on failure
    */
    uint8_t Measure(bool increment);

    /// Resets Device
    ///@return true on succes, false on failure
    bool reset();

    /// Get temperature of chip
    ///@return temperature in degrees Celcius
    float getTemperature();

    /// Puts the Device in Standby Mode
    ///@return true on succes, false on failure
    bool standby();

    /// Puts the Device in PowerDown Mode
    ///@return true on succes, false on failure
    bool powerdown();

    /** Initialises a frequency sweep.
    @param startFreq initial frequency (max 100000Hz)
    @param stepFreq stepsize of increment in frequency
    @param nrOfSteps number of increments
    @param nrOfCycles Number of cycles (outputfrequency) to wait for sampling can start  (max 2044)
    @param PGA Gain of the pre-amplifier (true = x1; false = x5)
    @param RangeNr Set the output voltage (1 = 2V; 2 = 1V; 3 = 400mV; 4 = 200mV)
    @return true on succes, false on failure
    */
    bool initFrequencySweepParam(unsigned int startFreq, unsigned int stepFreq, unsigned int nrOfSteps, unsigned int nrOfCycles, bool PGA, int RangeNr);

    /// real part of impedance (uncalibrated)
    int real;

    /// imaginary part of impedance (uncalibrated)
    int imaginary;

private:
    I2C sCom;
    uint8_t PGAandVoltout;

    bool setFrequencySweepParam(unsigned int startFreq, unsigned int stepFreq, unsigned int nrOfSteps);
    bool gotoAdressPointer(uint8_t Adress);
    bool setRegister(uint8_t RegisterAdress, uint8_t RegisterValue);
    bool writeBlock(uint8_t ByteArray[], uint8_t sizeArray);
    uint8_t getRegister(uint8_t RegisterAdress);
    bool readBlock(uint8_t* ByteArray, uint8_t sizeArray);
    bool setControlReg(uint8_t Command);
    uint8_t getData();
    bool _extClk;

};
#endif