#include "usbconfig.h"
#ifdef USB_USE_MALLOC
#include "mbed.h"
#define __DEBUG
#include "mydbg.h"
#include "usb_mem.h"
#include "UsbInc.h"

#define HCCA_SIZE 0x100
#define ED_SIZE 0x10
#define TD_SIZE 0x10
#define ITD_SIZE 0x20
#define UTD_SIZE (32+16)
#define BP_SIZE  (192*4)

static volatile byte* usb_hcca;  //256 bytes aligned!

inline static void* malloc_align(size_t size, size_t alignment)
{
    void* p;
    int r = posix_memalign(&p, alignment, size);
    if (r == 0) {
        return p;
    }
    return NULL;
}

static void utd_init()
{
    DBG_ASSERT(sizeof(HCTD) == 16);
    DBG_ASSERT(sizeof(HCITD) == 32);
    DBG_ASSERT(sizeof(HCUTD) == 48);
}

static void pb_init()
{
}

void usb_mem_init()
{
    usb_hcca = (uint8_t*)malloc_align(HCCA_SIZE, 256);

    utd_init();
    pb_init();

    DBG("--- Memory Map --- \n");
    DBG("usb_hcca       =%p\n", usb_hcca);
    DBG_ASSERT(((uint32_t)usb_hcca % 256) == 0);
}

volatile byte* usb_get_hcca()
{
    return usb_hcca;
}

volatile byte* usb_get_ed()
{
    return (uint8_t*)malloc_align(ED_SIZE, 16);
}

static uint8_t* usb_get_utd(int al)
{
    return (uint8_t*)malloc_align(UTD_SIZE, al);
}

volatile byte* usb_get_td(uint32_t endpoint)
{
    DBG_ASSERT(endpoint);
    uint8_t* td = usb_get_utd(16);
    if (td) {
        HCUTD* utd = (HCUTD*)td;
        memset(utd, 0x00, sizeof(HCTD));
        utd->type = 1;
        utd->UsbEndpoint = endpoint;
    }
    return td;
}

volatile byte* usb_get_itd(uint32_t endpoint)
{
    DBG_ASSERT(endpoint);
    uint8_t* itd = usb_get_utd(32);
    if (itd) {
        HCUTD* utd = (HCUTD*)itd;
        memset(utd, 0x00, sizeof(HCITD));
        utd->type = 2;
        utd->UsbEndpoint = endpoint;
    }
    return itd;
}

volatile byte* usb_get_bp(int size)
{
    DBG_ASSERT(size >= 128 && size <= BP_SIZE);
    if (size > BP_SIZE) {
        return NULL;
    }
    return (byte*)malloc(size);
}

int usb_bp_size()
{
    return BP_SIZE; 
}

void usb_free_ed(volatile byte* ed)
{
    free((void*)ed);
}

static void usb_free_utd(volatile uint8_t* utd)
{
    free((void*)utd);
}

void usb_free_td(volatile byte* td)
{
    usb_free_utd(td);
    return;
}

void usb_free_itd(volatile byte* itd)
{
    usb_free_utd(itd);
    return;
}

void usb_free_bp(volatile byte* bp)
{
    free((void*)bp);
}

bool usb_is_td(volatile byte* td)
{
    DBG_ASSERT(td);
    HCUTD* utd = (HCUTD*)td;
    DBG_ASSERT(utd->type != 0);
    return utd->type == 1;
}

bool usb_is_itd(volatile byte* itd)
{
    DBG_ASSERT(itd);
    HCUTD* utd = (HCUTD*)itd;
    DBG_ASSERT(utd->type != 0);
    return utd->type == 2;
}
#endif // USB_USE_MALLOC
