#include "mbed.h"
#include "usb_itd.h"
//#define __DEBUG
#include "mydbg.h"

usb_itd::usb_itd(HCITD* itd)
{
    DBG_ASSERT(itd);
    m_itd = itd;
}

bool usb_itd::Done()
{
    int cc = (m_itd->Control >> 28);
    if (cc == 0xe || cc == 0xf) { // not accessed?
        return false;
    }
    if (m_itd->Next) {
        return false;
    }
    return true;
}

int usb_itd::ConditionCode()
{
    int cc = (m_itd->Control >> 28);
    return cc;
}

uint16_t usb_itd::get_psw(int n)
{
    DBG_ASSERT(n >= 0 && n <= 7);
    DBG_ASSERT(m_itd);
    uint16_t psw = 0;
    switch(n) {
        case 0: psw = m_itd->OffsetPSW10;       break;
        case 1: psw = m_itd->OffsetPSW10 >> 16; break;
        case 2: psw = m_itd->OffsetPSW32;       break;
        case 3: psw = m_itd->OffsetPSW32 >> 16; break;
        case 4: psw = m_itd->OffsetPSW54;       break;
        case 5: psw = m_itd->OffsetPSW54 >> 16; break;
        case 6: psw = m_itd->OffsetPSW76;       break;
        case 7: psw = m_itd->OffsetPSW76 >> 16; break;
    }
    return psw;
}

uint16_t usb_itd::StartingFrame()
{
    return (m_itd->Control & ITD_SF);
}

int usb_itd::FrameCount()
{
    DBG_ASSERT(m_itd);
    int fc = ((m_itd->Control & ITD_FC) >> 24) + 1;
    DBG_ASSERT(fc >= 1 && fc <= 8); 
    return fc; 
}

int usb_itd::PacketStatus(int n)
{
    DBG_ASSERT(n >=0 && n <= 7);
    uint16_t psw = get_psw(n);
    int cc = (psw >> 12) & 0xf;
    return cc;
}

int usb_itd::Length(int n)
{
    DBG_ASSERT(n >=0 && n <= 7);
    uint16_t psw = get_psw(n);
    int size = psw & 0x7ff;
    int cc = (psw >> 12) & 0xf;
    if (cc == 0xe || cc == 0xf) {
        return -1;
    }
    if (cc == 0 || cc == 9) {
        return size;
    }
    debug();
    return -1;
}

uint8_t* usb_itd::BufferPage(int n, int size)
{
    DBG_ASSERT(n >=0 && n <= 7);
    DBG_ASSERT(size >= 128);
    DBG_ASSERT(size <= 1020);
    uint8_t* buf = (uint8_t*)m_itd->BufferPage0 + n * size;
    DBG_ASSERT((uint32_t)(buf+size-1) <= m_itd->BufferEnd);
    return buf;
}

void usb_itd::free()
{
    DBG_ASSERT(m_itd);
    if (m_itd) {
        uint8_t* buf = (uint8_t*)m_itd->BufferPage0;
        DBG_ASSERT(buf);
        usb_free_bp(buf);
        usb_free_itd((byte*)m_itd);
        m_itd = NULL;
    }
}

void usb_itd::debug()
{
    DBG("itd             =%08X\n", m_itd);
    DBG("itd->Control    =%08X\n", m_itd->Control);
    DBG("itd->BufferPage0=%08X\n", m_itd->BufferPage0); 
    DBG("itd->Next       =%08X\n", m_itd->Next); 
    DBG("itd->BufferEnd  =%08X\n", m_itd->BufferEnd); 
    DBG("itd->OffsetPSW10=%08X\n", m_itd->OffsetPSW10);
    DBG("itd->OffsetPSW32=%08X\n", m_itd->OffsetPSW32);
    DBG("itd->OffsetPSW54=%08X\n", m_itd->OffsetPSW54);
    DBG("itd->OffsetPSW76=%08X\n", m_itd->OffsetPSW76);
}
