// mydebug.cpp 2014/6/22
#include "mydebug.h"
#ifdef MY_DEBUG
#include "mbed.h"

struct trace_info {
    uint32_t timestamp_ms;
    const char* func;
    int line;
    uint32_t value;
};
    
struct StTrace {
    static const int TRACE_MAX = 32;
    trace_info buf[TRACE_MAX];
    int pos;
    int count;
    Timer t;
    StTrace() {
        clear();
        t.reset();
        t.start();
    }
    void clear() {
        pos = 0;
        count = 0;
    }
    void view() {
        fprintf(stderr, "TRACE:\n");
        int n = pos;
        for(int i = 0; i < TRACE_MAX && i < count; i++) {
            if (--n < 0) {
                n = TRACE_MAX-1;
            }
            fprintf(stderr, "%d %s@%d %p\n", buf[n].timestamp_ms, buf[n].func, buf[n].line, buf[n].value);
        }
    }
    void put(const char* func, int line, uint32_t value = 0) {
        buf[pos].timestamp_ms = t.read_ms();
        buf[pos].func = func;
        buf[pos].line = line;
        buf[pos].value = value;
        if (++pos >= TRACE_MAX) {
            pos = 0;
        }
        count++;
    }    
};

StTrace trace;

void mytrace(const char* func, int line) {
    trace.put(func, line);
}

void mytrace1(const char* func, int line, uint32_t value) {
    trace.put(func, line, value);
}

void mytrace_view() {
    trace.view();
}
      
void myassert(const char* func, int line, const char* expr) {
    fprintf(stderr, "\n\n%s@%d %s ASSERT!!!\n\n", func, line, expr);
    trace.view();
    trace.clear();
    exit(1);
}

void debug_hex(uint8_t* buf, int len)
{
    for(int i = 0; i < len; i++) {
        if ((i%16) == 0) {
            debug("+%04x:", i);
        }
        debug(" %02x", buf[i]);
        if ((i%16) == 15) {
            debug("\n");
        }
    }
    debug("\n");
}

void debug_dap_request(uint8_t* buf, int len)
{
    debug("uint8_t req[] = ");
    for(int i = 0; i < len; i++) {
        debug("%c0x%02x", i == 0 ? '{' : ',', buf[i]);
    }
    debug("};\n");
    char* cmd;
    char* param = "";
    char tmp[128];
    uint32_t count;
    int disp_hex = 0;
    switch(buf[0]) {
        case 0x00:
            cmd = "Info";
            switch(buf[1]) {
                case 1:    param = "VENDOR"; break;
                case 2:    param = "PRODUCT"; break;
                case 3:    param = "SER_NUM"; break;
                case 4:    param = "FW_VER"; break;
                case 0xfe: param = "PACKET_COUNT"; break;
                case 0xff: param = "PACKET_SIZE"; break;
                default:   param = "???"; break;
            }
            break;
        case 0x01:
            cmd = "LED";
            param = tmp;
            switch(buf[1]) {
                case 0: strcpy(tmp, "DEBUGGER_CONNECTED"); break;
                case 1: strcpy(tmp, "TARGET_RUNNING"); break;
                default: strcpy(tmp, "???"); break;
            }
            strcat(tmp, (buf[2]&0x01) ? " ON" : " OFF"); 
            break;
        case 0x02:
            cmd = "Connect";
            switch(buf[1]) {
                case 0: param = "AUTODETECT"; break;
                case 1: param = "SWD"; break;
                case 2: param = "JTAG"; break;
                default: param = "???"; break;
            }
            break;
        case 0x03: cmd = "Disconnect"; break;
        case 0x04: cmd = "TransferConfigure"; break;
        case 0x05:
            cmd = "Transfer";
            disp_hex = 16;
            break;
        case 0x06:
            cmd = "TransferBlock";
            disp_hex = 16;
            break;
        case 0x07: cmd = "TransferAbort"; break;
        case 0x08: cmd = "WriteABORT"; break;
        case 0x09: cmd = "Delay"; break;
        case 0x0a: cmd = "ResetTarget"; break;
        case 0x10:
            cmd = "SWJ_Pins";
            //snprintf(tmp, sizeof(tmp), "wait: %d us", buf[3]|buf[4]<<8|buf[5]<<16|buf[6]<<24);
            if (buf[2] & 0x01) {
                strcat(tmp, " SWCLK_TCK ");
                strcat(tmp, (buf[1] & 0x01) ? "ON" : "OFF");
            }
            if (buf[2] & 0x02) {
                strcat(tmp, " SWDIO_TMS ");
                strcat(tmp, (buf[1] & 0x02) ? "ON" : "OFF");
            }
            if (buf[2] & 0x04) {
                strcat(tmp, " TDI ");
                strcat(tmp, (buf[1] & 0x04) ? "ON" : "OFF");
            }
            if (buf[2] & 0x08) {
                strcat(tmp, " TDO ");
                strcat(tmp, (buf[1] & 0x08) ? "ON" : "OFF");
            }
            if (buf[2] & 0x20) {
                strcat(tmp, " nTRST ");
                strcat(tmp, (buf[1] & 0x20) ? "ON" : "OFF");
            }
            if (buf[2] & 0x80) {
                strcat(tmp, " nRESET ");
                strcat(tmp, (buf[1] & 0x80) ? "ON" : "OFF");
            }
            param = tmp;
            break;
        case 0x11: 
            cmd = "SWJ_Clock";
            //snprintf(tmp, sizeof(tmp), "%d Hz", buf[1]|buf[2]<<8|buf[3]<<16|buf[4]<<24);
            param = tmp;
            break;
        case 0x12:
            cmd = "SWJ_Sequence";
            count = buf[1];
            //snprintf(tmp, sizeof(tmp), "count: %d", count);
            param = tmp;
            break;
        case 0x13:
            cmd = "SWD_Configure";
            //snprintf(tmp, sizeof(tmp), "turnaround: %d, data_phase: %d", (buf[1]&0x03)+1, (buf[1]&0x04) ? 1 : 0); 
            param = tmp;
            break;
        case 0x14:
            cmd = "JTAG_Sequence";
            disp_hex = 16;
            break;
        default:
            cmd = "???";
            disp_hex = 16;
            break;
    }
    debug("DAP: %s %s ", cmd, param);
    for(int i = 0; i < disp_hex; i++) {
        debug("%02x ", buf[i]);
    }
    debug("\n");
}

void debug_dap_response(uint8_t* buf, int len)
{
    debug("uint8_t res[] = ");
    for(int i = 0; i < len; i++) {
        debug("%c0x%02x", i == 0 ? '{' : ',', buf[i]);
    }
    debug("};\n");
}

void debug_swd_transfer(uint8_t request, uint32_t wdata, uint32_t rdata, uint8_t ack)
{
    debug("SWD: ");
    if (ack != 0x01) {
        debug("ERR req=%02x wdata=%08x rdata=%08x ack=%02x\n", request, wdata, rdata, ack); 
        return;
    }
    switch(request) {
        case 0x00: debug("DP_ABORT %08x",       wdata); break;
        case 0x01: debug("AP_CSW W %08x",       wdata); break;
        case 0x02: debug("DP_IDCODE %08x",      rdata); break;
        case 0x03: debug("AP_CSW R %08x",       rdata); break;
        case 0x04: debug("DP_CTRL_STAT W %08x", wdata); break;
        case 0x05: debug("AP_TAR W %08x",       wdata); break;
        case 0x06: debug("DP_CTRL_STAT R %08x", rdata); break;
        case 0x07: debug("AP_TAR R %08x",       rdata); break;
        case 0x08: debug("DP_SELECT %08x",      wdata); break;
        case 0x0a: debug("DP_RESEND %08x",      rdata); break;
        //case 0x0b: debug("DP_RDBUFF W %08x",    wdata); break;
        case 0x0d: debug("AP_DRW W %08x",       wdata); break;
        case 0x0e: debug("DP_RDBUFF %08x",      rdata); break;
        case 0x0f: debug("AP_DRW R %08x",       rdata); break;
        default:
            debug("request=%02x wdata=%08x rdata=%08x ack=%02x", request, wdata, rdata, ack); 
            break; 
    }
    debug(" request=%02x(%s %s %x) wdata=%08x rdata=%08x ack=%02x", request, 
                                                                 request&1?"AP":"DP", request&2?"R":"nW",
                                                                        request&0xfc,
                                                                        wdata, rdata, ack); 
    debug("\n");
}

#endif
