// MyNetTcpSocket.cpp 2013/3/24
#include "mbed.h"
#include "w5200.h"
#include "MyNetTcpSocket.h"
//#define DEBUG
//#define DEBUG_LISTEN
#include "w5200debug.h"

#define USE_NO_DELAYED_ACK 1

int w5200_new_socket() 
{
    for(int i = 0; i < MAX_SOCK_NUM; i++) {
        if (W5200.readSnMR(i) == SnMR::CLOSE) { // 0x00
            if (W5200.readSnSR(i) != SnSR::LISTEN) { // 0x14
                return i;
            }
        }
    }
    return -1; // not found
}

MyNetTcpSocket::MyNetTcpSocket(int socket) : NetTcpSocket(),_socket(socket),wait_accept(false)  {
    DBG("socket: %d\n", socket);
    if (_socket == (-1)) {
        _socket = w5200_new_socket();
    }
    DBG("%p socket: %d\n", this, _socket);
    if (_socket != (-1)) {
        uint8_t Sn_MR = SnMR::TCP; // set TCP mode
#if USE_NO_DELAYED_ACK 
        Sn_MR |= SnMR::ND;
#endif 
        W5200.writeSnMR(_socket, Sn_MR);
    }
}

MyNetTcpSocket::~MyNetTcpSocket() {
    DBG("socket=%d\n", _socket);
    close();
    if (_socket != (-1)) {
        W5200.writeSnMR(_socket, SnMR::CLOSE);
    }
}

NetTcpSocketErr MyNetTcpSocket::bind(const Host& me) {
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    int port = me.getPort();
    W5200.writeSnPORT(_socket, port);
    return NETTCPSOCKET_OK;
}

NetTcpSocketErr MyNetTcpSocket::listen() {
    DBG("%p\n", this);
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    W5200.execCmdSn(_socket, Sock_OPEN); // set OPEN command
    W5200.execCmdSn(_socket, Sock_LISTEN); // listen
#ifdef DEBUG_LISTEN
    printf("[%s@%d]\n",__PRETTY_FUNCTION__,__LINE__);
    uint8_t ip[4];
    printf("socket:%d SnMR:%02x SnIR:%02x SnSR:%02x", _socket, 
        W5200.readSnMR(_socket), W5200.readSnIR(_socket), W5200.readSnSR(_socket));
    W5200.getIPAddress(ip);
    printf(" SIPR: %d.%d.%d.%d Sn_PORT:%d\n", ip[0], ip[1], ip[2], ip[3], W5200.readSnPORT(_socket));
#endif //DEBUG_LISTEN
    wait_accept = true;
    return NETTCPSOCKET_OK;
}

NetTcpSocketErr MyNetTcpSocket::connect(const Host& host) {
    DBG("%p host=%p\n", this, host);
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    uint8_t ip[4];
    ip[0] = host.getIp()[0];
    ip[1] = host.getIp()[1];
    ip[2] = host.getIp()[2];
    ip[3] = host.getIp()[3];
    int port = host.getPort();
    W5200.writeSnDIPR(_socket, ip);
    W5200.writeSnDPORT(_socket, port);
    if (W5200.readSnPORT(_socket) == 0) {
        W5200.writeSnPORT(_socket, 1024 + _socket);
    }
    W5200.execCmdSn(_socket, Sock_OPEN); // set OPEN command
    W5200.execCmdSn(_socket, Sock_CONNECT);
#ifdef DEBUG_CONNECT
    printf("socket:%d SnMR:%02x SnIR:%02x SnSR:%02x\n", _socket, 
        W5200.readSnMR(_socket), W5200.readSnIR(_socket), W5200.readSnSR(_socket));
    W5200.getIPAddress(ip);
    printf("SIPR: %d.%d.%d.%d Sn_PORT:%d\n", ip[0], ip[1], ip[2], ip[3], W5200.readSnPORT(_socket));
    W5200.readSnDIPR(_socket, ip);
    printf("Sn_DIPR: %d.%d.%d.%d Sn_DPORT:%d\n", ip[0], ip[1], ip[2], ip[3], W5200.readSnDPORT(_socket));
#endif //DEBUG_CONNECT
    return NETTCPSOCKET_OK;
}

NetTcpSocketErr MyNetTcpSocket::accept(Host* pClient, NetTcpSocket** ppNewNetTcpSocket) {
    DBG("%p pClient=%p\n", this, pClient);
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    uint8_t ip[4];
    W5200.readSnDIPR(_socket, ip);
    pClient->setIp(IpAddr(ip[0],ip[1],ip[2],ip[3]));
    int port = W5200.readSnDPORT(_socket);
    pClient->setPort(port);
    Host me;
    me.setPort(W5200.readSnPORT(_socket));
    MyNetTcpSocket* pNewNetTcpSocket = new MyNetTcpSocket(_socket);
    if (pNewNetTcpSocket == NULL) {
        return NETTCPSOCKET_EMPTY;
    }
    pNewNetTcpSocket->m_refs++;
    *ppNewNetTcpSocket = pNewNetTcpSocket;
    _socket = w5200_new_socket();
    if (_socket != (-1)) {
        uint8_t Sn_MR = SnMR::TCP; // set TCP mode
#if USE_NO_DELAYED_ACK 
        Sn_MR |= SnMR::ND;
#endif 
        W5200.writeSnMR(_socket, Sn_MR);
        bind(me);
        listen();
     }
    return NETTCPSOCKET_OK;
}

int /*if < 0 : NetTcpSocketErr*/ MyNetTcpSocket::send(const char* buf, int len) {
    DBG_STR((uint8_t*)buf,len);
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    if (len > 0) {
        W5200.send_data_processing(_socket, (uint8_t*)buf, len);
        W5200.execCmdSn(_socket, Sock_SEND);
    }
    return len;
}

int /*if < 0 : NetTcpSocketErr*/ MyNetTcpSocket::recv(char* buf, int len){
    if (_socket == (-1)) {
        return NETTCPSOCKET_MEM;
    }
    int size = W5200.getRXReceivedSize(_socket);
    if (size > len) {
        size = len;
    }
    if (size > 0) {
        W5200.recv_data_processing(_socket, (uint8_t*)buf, size);
        W5200.execCmdSn(_socket, Sock_RECV);
    }
    DBG_STR((uint8_t*)buf, size);
    return size;
}

NetTcpSocketErr MyNetTcpSocket::close() {
    DBG("%p m_closed=%d m_refs=%d\n", this, m_closed, m_refs);
    if(m_closed) {
        return NETTCPSOCKET_OK;
    }
    m_closed = true;
    cleanUp();
    if (_socket != (-1)) {
        W5200.execCmdSn(_socket, Sock_DISCON);
        W5200.execCmdSn(_socket, Sock_CLOSE);
    }
    return NETTCPSOCKET_OK;
}

NetTcpSocketErr MyNetTcpSocket::poll(){
    NetTcpSocket::flushEvents();
#ifdef DEBUG_POLL
    printf("%p socket:%d\n", this,_socket);
    if (_socket != (-1)) {
        printf("SnMR:%02x SnIR:%02x SnSR:%02x\n", 
            W5200.readSnMR(_socket), W5200.readSnIR(_socket), W5200.readSnSR(_socket));
        uint8_t ip[4];
        W5200.readSnDIPR(_socket, ip);
        printf("Sn_DIPR: %d.%d.%d.%d Sn_DPORT: %d\n", ip[0], ip[1], ip[2], ip[3], W5200.readSnDPORT(_socket));
        printf("Sn_RX_RSR:%5d, Sn_RX_RD:%5d, Sn_RX_WR:%5d\n",
                W5200.readSnRX_RSR(_socket), W5200.readSnRX_RD(_socket), W5200.readSnRX_WR(_socket));
        printf("Sn_TX_FSR:%5d, Sn_TX_RD:%5d, Sn_TX_WR:%5d\n",
                W5200.readSnTX_FSR(_socket), W5200.readSnTX_RD(_socket), W5200.readSnTX_WR(_socket));
    }
    wait_ms(200);
#endif //DEBUG_POLL
    if (_socket == (-1)) {
        return NETTCPSOCKET_OK;
    }
    uint8_t Sn_SR = W5200.readSnSR(_socket);
    if (wait_accept) {
        if (Sn_SR == 0x17) {
            queueEvent(NETTCPSOCKET_ACCEPT);
            wait_accept = false;
        }
    }
    if (Sn_SR == 0x1c) {
        queueEvent(NETTCPSOCKET_CONRST);
    }
    if (W5200.getRXReceivedSize(_socket) > 0) {
        queueEvent(NETTCPSOCKET_READABLE);
    }
    if (Sn_SR == 0x17) {
        queueEvent(NETTCPSOCKET_CONNECTED);
        if (W5200.getTXFreeSize(_socket) > 0) {
            queueEvent(NETTCPSOCKET_WRITEABLE);
        }
    }
    if (Sn_SR == 0x00) {
        queueEvent(NETTCPSOCKET_DISCONNECTED);
    }
    return NETTCPSOCKET_OK;
}

void MyNetTcpSocket::cleanUp() //Flush input buffer
{
    if (_socket == (-1)) {
        return;
    }
    while(W5200.getRXReceivedSize(_socket) > 0) {
        uint8_t temp[1];
        W5200.recv_data_processing(_socket, temp, 1);
        W5200.execCmdSn(_socket, Sock_RECV);
    }    
}
