/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "stdint.h"
#include "USB_HID.h"

USB_HID::USB_HID(USBDevice* device, uint8_t output_report_length, uint8_t input_report_length) : _device(device)
{
    output_length = output_report_length;
    input_length = input_report_length;
}

bool USB_HID::send(HID_REPORT *report)
{
    return _device->write(HID_EPINT_IN, report->data, report->length, MAX_HID_REPORT_SIZE);
}

bool USB_HID::sendNB(HID_REPORT *report)
{
    return _device->writeNB(HID_EPINT_IN, report->data, report->length, MAX_HID_REPORT_SIZE);
}

bool USB_HID::read(HID_REPORT *report)
{
    uint32_t bytesRead = 0;
    bool result;
    result = _device->readEP(HID_EPINT_OUT, report->data, &bytesRead, MAX_HID_REPORT_SIZE);
    if(!_device->readStart(HID_EPINT_OUT, MAX_HID_REPORT_SIZE))
        return false;
    report->length = bytesRead;
    return result;
}

bool USB_HID::readNB(HID_REPORT *report)
{
    uint32_t bytesRead = 0;
    bool result;
    result = _device->readEP_NB(HID_EPINT_OUT, report->data, &bytesRead, MAX_HID_REPORT_SIZE);
    report->length = bytesRead;
    if(!_device->readStart(HID_EPINT_OUT, MAX_HID_REPORT_SIZE))
        return false;
    return result;
}

/* virtual */ uint8_t * USB_HID::reportDesc() {
    static uint8_t reportDescriptor[] = {
        0x06, 0x00, 0xff,
        0x09, 0x01,         // usage
        0xA1, 0x01,         // Collection 0x01
        0x15, 0x00,         // logical minimum = 0
        0x26, 0xFF, 0x00,   // logical maximum = 255
        0x75, 0x08,         // report size = 8 bits
        0x95, 0x40,         // report count
        0x09, 0x01,         // usage
        0x81, 0x02,         // Input (array)
        0x95, 0x40,         // report count
        0x09, 0x01,         // usage
        0x91, 0x02,         // Output (array)
        0x95, 0x01,         // report count
        0x09, 0x01,         // usage
        0xb1, 0x02,
        0xC0                // end collection
    };
    reportLength = sizeof(reportDescriptor);
    return reportDescriptor;
}

/* virtual */ uint16_t USB_HID::reportDescLength() {
    reportDesc();
    return reportLength;
}

bool USB_HID::Request_callback(CONTROL_TRANSFER* transfer, uint8_t* hidDescriptor)
{
    // Process additional standard requests
    if (transfer->setup.bmRequestType.Type == STANDARD_TYPE) {
        switch (transfer->setup.bRequest) {
            case GET_DESCRIPTOR:
                switch (DESCRIPTOR_TYPE(transfer->setup.wValue)) {
                    case REPORT_DESCRIPTOR:
                        if ((reportDesc() != NULL) && (reportDescLength() != 0)) {
                            transfer->remaining = reportDescLength();
                            transfer->ptr = reportDesc();
                            transfer->direction = DEVICE_TO_HOST;
                            return true;
                        }
                        break;
                    case HID_DESCRIPTOR:
                        if (hidDescriptor != NULL) {
                            transfer->remaining = HID_DESCRIPTOR_LENGTH;
                            transfer->ptr = hidDescriptor;
                            transfer->direction = DEVICE_TO_HOST;
                            return true;
                        }
                        break;
                    default:
                        break;
                }
                break;
            default:
                break;
        }
    }

    if (transfer->setup.bmRequestType.Type == CLASS_TYPE) {
        switch (transfer->setup.bRequest) {
             case SET_REPORT:
                // First byte will be used for report ID
                outputReport.data[0] = transfer->setup.wValue & 0xff;
                outputReport.length = transfer->setup.wLength + 1;

                transfer->remaining = sizeof(outputReport.data) - 1;
                transfer->ptr = &outputReport.data[1];
                transfer->direction = HOST_TO_DEVICE;
                transfer->notify = true;
                return true;
        }
    }
    return false;
}
