/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "stdint.h"
#include "USB_CDC.h"
#define MY_DEBUG
#include "mydebug.h"

#define CDC_SET_LINE_CODING        0x20
#define CDC_GET_LINE_CODING        0x21
#define CDC_SET_CONTROL_LINE_STATE 0x22
#define CDC_SEND_BREAK             0x23

#define MAX_CDC_REPORT_SIZE MAX_PACKET_SIZE_EPBULK

USB_CDC::USB_CDC(USBDevice* device) : _device(device), _rx_buf(128)
{
    terminal_connected = false;
    //USBDevice::connect();
}

void USB_CDC::putc(int c)
{
    if (terminal_connected) {
        uint8_t buf[1];
        buf[0] = c;
        _device->write(CDC_EPBULK_IN, buf, sizeof(buf), MAX_CDC_REPORT_SIZE);
    }
}

int USB_CDC::getc()
{
    uint8_t c = 0;
    while (_rx_buf.isEmpty());
    _rx_buf.dequeue(&c);
    return c;
}

int USB_CDC::readable()
{
    return _rx_buf.available() > 0 ? 1 : 0;
}

int USB_CDC::writeable()
{
    return 1;
}

void USB_CDC::baud_callback(int baudrate)
{
    DBG("baudrate=%d", baudrate);
}

void USB_CDC::send_break_callback(uint16_t duration)
{
    DBG("duration=%04x", duration);
}

void USB_CDC::control_line_callback(int rts, int dtr)
{
    DBG("rts=%d, dtr=%d", rts, dtr);
}

bool USB_CDC::send(uint8_t * buffer, uint32_t size) {
    return _device->write(CDC_EPBULK_IN, buffer, size, MAX_CDC_REPORT_SIZE);
}

bool USB_CDC::readEP(uint8_t * buffer, uint32_t * size) {
    if (!_device->readEP(CDC_EPBULK_OUT, buffer, size, MAX_CDC_REPORT_SIZE))
        return false;
    if (!_device->readStart(CDC_EPBULK_OUT, MAX_CDC_REPORT_SIZE))
        return false;
    return true;
}

bool USB_CDC::readEP_NB(uint8_t * buffer, uint32_t * size) {
    if (!_device->readEP_NB(CDC_EPBULK_OUT, buffer, size, MAX_CDC_REPORT_SIZE))
        return false;
    if (!_device->readStart(CDC_EPBULK_OUT, MAX_CDC_REPORT_SIZE))
        return false;
    return true;
}

bool USB_CDC::Request_callback(CONTROL_TRANSFER* transfer)
{
    static uint8_t cdc_line_coding[7]= {0x80, 0x25, 0x00, 0x00, 0x00, 0x00, 0x08};

    if (transfer->setup.bmRequestType.Type == CLASS_TYPE) {
        switch (transfer->setup.bRequest) {
            case CDC_SET_LINE_CODING: // 0x20
                transfer->remaining = 7;
                transfer->notify = true;
                terminal_connected = true;
                return true;

            case CDC_GET_LINE_CODING: // x021
                transfer->remaining = 7;
                transfer->ptr = cdc_line_coding;
                transfer->direction = DEVICE_TO_HOST;
                return true;

            case CDC_SET_CONTROL_LINE_STATE: // 0x22
                control_line_callback((transfer->setup.wValue>>1) & 1, (transfer->setup.wValue) & 1);
                terminal_connected = false;
                return true;
            
            case CDC_SEND_BREAK: // 0x23
                send_break_callback(transfer->setup.wValue);
                return true;
        }
    }
    return false;
}

static uint32_t LD32(uint8_t* buf)
{
    return buf[0]|buf[1]<<8|buf[2]<<16|buf[3]<<24;
}

bool USB_CDC::RequestCompleted_callback(CONTROL_TRANSFER* transfer, uint8_t* buf, int length)
{
    int baudrate;
    if (transfer->setup.bmRequestType.Type == CLASS_TYPE) {
        switch (transfer->setup.bRequest) {
            case CDC_SET_LINE_CODING: // 0x20
                baudrate = LD32(buf);
                baud_callback(baudrate);
                return true;
        }
    }
    DBG_HEX((uint8_t*)transfer, sizeof(CONTROL_TRANSFER));
    return false;           
}

bool USB_CDC::EPBULK_OUT_callback() // virtual COM to target
{
    uint8_t buf[MAX_CDC_REPORT_SIZE];
    uint32_t size = 0;
    //we read the packet received and put it on the circular buffer
    _device->readEP(CDC_EPBULK_OUT, buf, &size, MAX_CDC_REPORT_SIZE);
    for(int i = 0; i < size; i++) {
        _rx_buf.queue(buf[i]);
    }

    // We reactivate the endpoint to receive next characters
    _device->readStart(CDC_EPBULK_OUT, MAX_PACKET_SIZE_EPBULK);
    return true;
}
