/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#pragma once

/* These headers are included for child class. */
#include "USBEndpoints.h"
#include "USBDescriptor.h"
#include "USBDevice_Types.h"

#include "USBHID_Types.h"
#include "USBDevice.h"

#define HID_EPINT_IN   EP4IN
#define HID_EPINT_OUT  EP4OUT

/** USB HID device for CMSIS-DAP
 */
class USB_HID {
public:

    /**
    * Constructor
    *
    * @param output_report_length Maximum length of a sent report (up to 64 bytes) (default: 64 bytes)
    * @param input_report_length Maximum length of a received report (up to 64 bytes) (default: 64 bytes)
    */
    USB_HID(USBDevice* device, uint8_t output_report_length = 64, uint8_t input_report_length = 64);


    /**
    * Send a Report. warning: blocking
    *
    * @param report Report which will be sent (a report is defined by all data and the length)
    * @returns true if successful
    */
    bool send(HID_REPORT *report);
    
    
    /**
    * Send a Report. warning: non blocking
    *
    * @param report Report which will be sent (a report is defined by all data and the length)
    * @returns true if successful
    */
    bool sendNB(HID_REPORT *report);
    
    /**
    * Read a report: blocking
    *
    * @param report pointer to the report to fill
    * @returns true if successful
    */
    bool read(HID_REPORT * report);
    
    /**
    * Read a report: non blocking
    *
    * @param report pointer to the report to fill
    * @returns true if successful
    */
    bool readNB(HID_REPORT * report);

    /*
    * Get the length of the report descriptor
    *
    * @returns the length of the report descriptor
    */
    virtual uint16_t reportDescLength();

    bool Request_callback(CONTROL_TRANSFER* transfer, uint8_t* hidDescriptor);
protected:
    /*
    * Get the Report descriptor
    *
    * @returns pointer to the report descriptor
    */
    virtual uint8_t * reportDesc();

    uint16_t reportLength;

private:
    USBDevice* _device;
    HID_REPORT outputReport;
    uint8_t output_length;
    uint8_t input_length;
};
