#pragma once
#include "Storage.h"

/** Access the LocalFileSystem using RamDisk(64KB)
 *
 * @code
 * #include "USBLocalFileSystem.h"
 *
 * int main() {
 *     USBLocalFileSystem* usb_local = new USBLocalFileSystem(); // RamDisk(64KB) 
 *
 *     FILE *fp = fopen("/local/mbed.txt", "a");
 *     fprintf(fp, "Hello World!\n");
 *     fclose(fp);
 * }
 * @endcode
 */
class USBLocalFileSystem {
public:

    /** Create the Local File System using RamDisk(64KB)
     *
     * @param name The name used to access the virtual filesystem
     */
    USBLocalFileSystem(const char* name = "local");

    /** Create the Local File System for accessing an SD Card using SPI
     *
     * @param mosi SPI mosi pin connected to SD Card
     * @param miso SPI miso pin conencted to SD Card
     * @param sclk SPI sclk pin connected to SD Card
     * @param cs   DigitalOut pin used as SD Card chip select
     * @param name The name used to access the virtual filesystem
     */    
    USBLocalFileSystem(PinName mosi, PinName miso, PinName sclk, PinName cs, const char* name = "local");
    
    /** Create the Local File System using StorageInterface
     *
     * @param storage StorageInterface 
     * @param name The name used to access the virtual filesystem
     */ 
    USBLocalFileSystem(StorageInterface* storage, const char* name = "local");

    void remount(); // remount local storage
    int lock(bool f);
    bool find(char* name, size_t size, const char* pat);

    /** Determine if there is a character available to read
     *
     *  @returns
     *    1 if there is a character available to read,
     *    0 otherwise
     */
    int readable();
    
     /** Determine if there is space available to write a character
     *
     *  @returns
     *    1 if there is space to write a character,
     *    0 otherwise
     */
    int writeable();

    /** Read a char from the serial port
     *
     * @returns The char read from the serial port
     */
    int getc();

    /** Write a char to the serial port
     *
     * @param c The char to write
     *
     * @returns The written char or -1 if an error occured
     */
    int putc(int c);

    /** Write a string to the serial port
     *
     * @param str The string to write
     *
     * @returns 0 if the write succeeds, EOF for error
     */
    int puts(const char* str);

    void attachSendBreak(void (*fptr)(uint16_t duration));
    void attachControlLineState(void (*fptr)(int dts, int dtr));
    void attachSettingChanged(void (*fptr)(int baud, int bits, int parity, int stop));

    StorageInterface* getStoage() { return _storage; }
    LocalStorage* getLocal() { return _local; }
    USBStorage2* getUsb() { return _usb; }

private:
    void init(StorageInterface* storage, const char* name);
    const char* _name;
    StorageInterface* _storage;
    LocalStorage* _local;
    USBStorage2* _usb;
};
