#if defined(TARGET_STM32F4)
/**    IAP : internal Flash memory access library
 *
 *        The internal Flash memory access is described in the LPC1768 and LPC11U24 usermanual.
 *            http://www.nxp.com/documents/user_manual/UM10360.pdf
 *            http://www.nxp.com/documents/user_manual/UM10462.pdf
 *
 *               LPC1768 --
 *                    Chapter  2: "LPC17xx Memory map"
 *                    Chapter 32: "LPC17xx Flash memory interface and programming"
 *                    refering Rev. 01 - 4 January 2010
 *
 *               LPC11U24 --
 *                    Chapter  2: "LPC11Uxx Memory mapping"
 *                    Chapter 20: "LPC11Uxx Flash programming firmware"
 *                    refering Rev. 03 - 16 July 2012
 *
 *        Released under the MIT License: http://mbed.org/license/mit
 *
 *        revision 1.0  09-Mar-2010   1st release
 *        revision 1.1  12-Mar-2010   chaged: to make possible to reserve flash area for user
 *                                            it can be set by USER_FLASH_AREA_START and USER_FLASH_AREA_SIZE in IAP.h
 *        revision 2.0  26-Nov-2012   LPC11U24 code added
 *        revision 2.1  26-Nov-2012   EEPROM access code imported from Suga koubou san's (http://mbed.org/users/okini3939/) library
 *                                            http://mbed.org/users/okini3939/code/M0_EEPROM_test/
 *        revision 3.0  09-Jan-2015   LPC812 and LPC824 support added
 *        revision 3.1  13-Jan-2015   LPC1114 support added
 *        revision 3.1.1 16-Jan-2015  Target MCU name changed for better compatibility across the platforms
 */

#include    "mbed.h"
#include    "IAP.h"

#define     USER_FLASH_AREA_START_STR( x )      STR( x )
#define     STR( x )                            #x

//unsigned char user_area[ USER_FLASH_AREA_SIZE ] __attribute__((section( ".ARM.__at_" USER_FLASH_AREA_START_STR( USER_FLASH_AREA_START ) ), zero_init));

/*
 *  Reserve of flash area is explained by Igor. Please refer next URL
 *    http://mbed.org/users/okano/notebook/iap-in-application-programming-internal-flash-eras/?page=1#comment-271
 */

//unsigned char user_area[ size ] __attribute__((section(".ARM.__at_0x78000"), zero_init));

/*
 *  IAP command codes
 *  Table 589. "IAP Command Summary", Chapter 8. "IAP commands", usermanual
 */

enum command_code {
    IAPCommand_Prepare_sector_for_write_operation    = 50,
    IAPCommand_Copy_RAM_to_Flash,
    IAPCommand_Erase_sector,
    IAPCommand_Blank_check_sector,
    IAPCommand_Read_part_ID,
    IAPCommand_Read_Boot_Code_version,
    IAPCommand_Compare,
    IAPCommand_Reinvoke_ISP,
    IAPCommand_Read_device_serial_number,
#if defined(TARGET_LPC11UXX)
    IAPCommand_EEPROM_Write = 61,
    IAPCommand_EEPROM_Read,
#elif defined(TARGET_LPC81X) || defined(TARGET_LPC82X)
    IAPCommand_Erase_page = 59,
#endif
};

int IAP::reinvoke_isp( void ) {
    return INVALID_COMMAND;
}

/** Read part identification number
 *
 *  @return    device ID
 *  @see       read_serial()
 */
int IAP::read_ID( void )
{
    IAP_result[1] = HAL_GetDEVID();
    return ( (int)IAP_result[ 1 ] );    //  to return the number itself (this command always returns CMD_SUCCESS)
}

int *IAP::read_serial( void ) {
    memcpy(&IAP_result[1], (void*)0x1fff7a10,  sizeof(uint32_t) * 3); // RM0383 24.1
    IAP_result[4] = 0; 
    return ( (int *)&IAP_result[ 1 ] );    //  to return the number itself (this command always returns CMD_SUCCESS)
}

int IAP::blank_check( int start, int end )
{
    if (start > end || !IS_FLASH_SECTOR(start) || !IS_FLASH_SECTOR(end)) {
        return INVALID_SECTOR;
    }
    static const int offset[8+1] = {0x00000, 0x04000, 0x08000, 0xc000, 0x10000, 0x20000, 0x40000, 0x60000, 0x80000};
    uint8_t* it = reinterpret_cast<uint8_t*>(FLASH_BASE + offset[start]);
    uint8_t* it_end = reinterpret_cast<uint8_t*>(FLASH_BASE + offset[end + 1]);
    for(; it != it_end; it++) {
        if (*it != 0xff) {
            return SECTOR_NOT_BLANK;
        }
    }
    return CMD_SUCCESS;
}

struct FLASH_Unlock {
    FLASH_Unlock() { HAL_FLASH_Unlock(); }
    ~FLASH_Unlock() { HAL_FLASH_Lock(); }
};

int IAP::erase( int start, int end )
{
    if (start > end || !IS_FLASH_SECTOR(start) || !IS_FLASH_SECTOR(end)) {
        return INVALID_SECTOR;
    }
    FLASH_EraseInitTypeDef Erase;
    Erase.TypeErase = FLASH_TYPEERASE_SECTORS;
    Erase.VoltageRange = FLASH_VOLTAGE_RANGE_1;
    Erase.NbSectors = end - start + 1;
    Erase.Sector = start;
    uint32_t PageError = 0;
    FLASH_Unlock unlock;
    HAL_StatusTypeDef status = HAL_FLASHEx_Erase(&Erase, &PageError);
    return status == HAL_OK ? CMD_SUCCESS : INVALID_COMMAND;
}

int IAP::prepare( int start, int end )
{
    if (start > end || !IS_FLASH_SECTOR(start) || !IS_FLASH_SECTOR(end)) {
        return INVALID_SECTOR;
    }
    return CMD_SUCCESS;
}

int IAP::write( char *source_addr, char *target_addr, int size )
{
    if (!IS_FLASH_ADDRESS((int)target_addr) || !IS_FLASH_ADDRESS((int)target_addr + size - 1)) {
        return DST_ADDR_NOT_MAPPED;
    }    
    uint32_t data;
    FLASH_Unlock unlock;
    for(int n = 0; n < size; n += sizeof(data)) {
        memcpy(&data, source_addr + n, sizeof(data));
        if (HAL_OK != HAL_FLASH_Program(TYPEPROGRAM_WORD, (uint32_t)target_addr + n, data)) {
            return INVALID_COMMAND;
        }
    }    
    return CMD_SUCCESS;            
}

int IAP::compare( char *source_addr, char *target_addr, int size )
{
    return memcmp(source_addr, target_addr, size) == 0 ? CMD_SUCCESS : COMPARE_ERROR;
}

int IAP::read_BootVer(void)
{
    // TODO
    return 0;
}

char * IAP::reserved_flash_area_start( void )
{
    // TODO
    return NULL;
}

int IAP::reserved_flash_area_size( void )
{
    // TODO
    return 0;
}

#if defined(TARGET_LPC11UXX)

int IAP::write_eeprom( char *source_addr, char *target_addr, int size )
{
    IAP_command[ 0 ]    = IAPCommand_EEPROM_Write;
    IAP_command[ 1 ]    = (unsigned int)target_addr;    //  Destination EEPROM address where data bytes are to be written. This address should be a 256 byte boundary.
    IAP_command[ 2 ]    = (unsigned int)source_addr;    //  Source RAM address from which data bytes are to be read. This address should be a word boundary.
    IAP_command[ 3 ]    = size;                         //  Number of bytes to be written. Should be 256 | 512 | 1024 | 4096.
    IAP_command[ 4 ]    = cclk_kHz;                     //  CPU Clock Frequency (CCLK) in kHz.

    iap_entry( IAP_command, IAP_result );

    return ( (int)IAP_result[ 0 ] );
}

int IAP::read_eeprom( char *source_addr, char *target_addr, int size )
{
    IAP_command[ 0 ]    = IAPCommand_EEPROM_Read;
    IAP_command[ 1 ]    = (unsigned int)source_addr;    //  Source EEPROM address from which data bytes are to be read. This address should be a word boundary.
    IAP_command[ 2 ]    = (unsigned int)target_addr;    //  Destination RAM address where data bytes are to be written. This address should be a 256 byte boundary.
    IAP_command[ 3 ]    = size;                         //  Number of bytes to be written. Should be 256 | 512 | 1024 | 4096.
    IAP_command[ 4 ]    = cclk_kHz;                     //  CPU Clock Frequency (CCLK) in kHz.

    iap_entry( IAP_command, IAP_result );

    return ( (int)IAP_result[ 0 ] );
}

#elif defined(TARGET_LPC81X) || defined(TARGET_LPC82X)

int IAP::erase_page( int start, int end )
{
    IAP_command[ 0 ]    = IAPCommand_Erase_page;
    IAP_command[ 1 ]    = (unsigned int)start;  //  Start Sector Number
    IAP_command[ 2 ]    = (unsigned int)end;    //  End Sector Number (should be greater than or equal to start sector number)
    IAP_command[ 3 ]    = cclk_kHz;             //  CPU Clock Frequency (CCLK) in kHz

    iap_entry( IAP_command, IAP_result );

    return ( (int)IAP_result[ 0 ] );
}

#endif

#endif // TARGET_STM32F4
