// GraphicOLED.h
#pragma once
#include "mbed.h"

/** GraphicOLED interface for WS0010
 *
 * Currently support UTF-8 KANJI(misaki font) 
 *
 * @code
 * #include "mbed.h"
 * #include "GraphicOLED.h"
 * 
 * GraphicOLED oled(p24, p26, p27, p28, p29, p30); // rs, e, d4-d7
 * 
 * int main() {
 *     oled.printf("Hello World!\n");
 * }
 * @endcode
 */
class GraphicOLED : public Stream {
public:
    /** Create a GraphicOLED interface
     *
     * @param rs    Instruction/data control line
     * @param e     Enable line (clock)
     * @param d4-d7 Data lines for using as a 4-bit interface
     */
    GraphicOLED(PinName rs, PinName e, PinName d4, PinName d5, PinName d6, PinName d7);

#if DOXYGEN_ONLY
    /** Write a character to the OLED
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formated string to the OLED
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif

    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** Clear the screen and locate to 0,0 */
    void cls();

    int rows();
    int columns();

    void g_write(uint8_t pat, int x, int y);
    void g_write(uint8_t *buf, int len, int x, int y);
       
private:
    int _uni_len;
    uint32_t _unicode;

    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

    int character(int column, int row, int c);
    void writeByte(uint8_t value);
    void writeCommand(uint8_t command);
    void writeData(uint8_t data);

    DigitalOut _rs, _e;
    BusOut _d;

    int _column;
    int _row;
};

