// BaseUsbHostHub.cpp 2013/1/25
#include "mbed.h"
#include "rtos.h"
#include "BaseUsbHost.h"
//#define DEBUG
#include "BaseUsbHostDebug.h"
#define TEST
#include "BaseUsbHostTest.h"

#define PORT_CONNECTION   0
#define PORT_ENABLE       1
#define PORT_SUSPEND      2
#define PORT_OVER_CURRENT 3
#define PORT_RESET        4
#define PORT_POWER        8
#define PORT_LOW_SPEED    9

#define C_PORT_CONNECTION   16
#define C_PORT_ENABLE       17
#define C_PORT_SUSPEND      18
#define C_PORT_OVER_CURRENT 19
#define C_PORT_RESET        20

UsbHub::UsbHub(ControlEp* ctlEp)
{
    if (ctlEp == NULL) {
        DBG_OHCI(LPC_USB->HcRhPortStatus1);
        int lowSpeed = 0;
        if (LPC_USB->HcRhPortStatus1 & 0x200) {
            lowSpeed = 1;
        }
        m_ctlEp = new ControlEp(lowSpeed);
        TEST_ASSERT_TRUE(m_ctlEp);
    } else {
        m_ctlEp = ctlEp;
    }
    CTASSERT(sizeof(StandardDeviceDescriptor) == 18);
    StandardDeviceDescriptor devdesc;
    int rc = m_ctlEp->GetDescriptor(1, 0, reinterpret_cast<uint8_t*>(&devdesc), sizeof(StandardDeviceDescriptor));
    TEST_ASSERT(rc == USB_OK);
    if (rc != USB_OK) {
        return;
    }
    TEST_ASSERT_TRUE(devdesc.bLength == 0x12);
    TEST_ASSERT_TRUE(devdesc.bDescriptorType == 0x01);
    TEST_ASSERT_TRUE(devdesc.bDeviceClass == 0x09); // hub
    if (devdesc.bDeviceClass != 0x09) { // hub ?
        return;
    }
    CTASSERT(sizeof(HubDescriptor) == 9);
    TEST_ASSERT(sizeof(HubDescriptor) == 9);
    HubDescriptor hubdesc;
    rc = m_ctlEp->controlReceive(0xa0, 6, 0x29<<8, 0, reinterpret_cast<uint8_t*>(&hubdesc), sizeof(HubDescriptor));
    TEST_ASSERT(rc == USB_OK);
    if (rc != USB_OK) {
        return;
    }
    TEST_ASSERT(hubdesc.bDescLength == 9); // length
    TEST_ASSERT(hubdesc.bDescriptorType == 0x29); // hub
    TEST_ASSERT(hubdesc.bNbrPorts >= 1);
    TEST_ASSERT(hubdesc.bNbrPorts <= 16);

    m_ctlEp->SetConfiguration(1);

    uint32_t status;
    rc = m_ctlEp->controlReceive(0xa0, 0, 0, 0, reinterpret_cast<uint8_t*>(&status), 4);
    TEST_ASSERT(rc == USB_OK);
    if (rc != USB_OK) {
        return;
    }
    DBG("HUB STATUS: %08X\n", status);

    for(int i = 1; i <= hubdesc.bNbrPorts; i++) {
        SetPortPower(i); // power on
        wait_ms(hubdesc.bPwrOn2PwrGood*2);
        uint32_t status;
        GetPortStatus(i, &status);
        DBG("port: %d status: %08X\n", i, status);
        if (status & 0x010000) { // Connect Status Change, has changed
            TEST_ASSERT(status & 0x000001);
            ClearPortFeature(C_PORT_CONNECTION, i);
            int lowSpeed = 0;
            if (status & 0x0200) {
                lowSpeed = 1;
            }
            PortReset(i);
            ControlEp* ctlEp = new ControlEp(lowSpeed);
            PortEp.push_back(ctlEp);
        } else {
            ClearPortPower(i); // power off
        }
    }
}

bool UsbHub::check(ControlEp* ctlEp)
{
    if (ctlEp == NULL) {
        return false;
    }
    StandardDeviceDescriptor desc;
    int rc = ctlEp->GetDescriptor(USB_DESCRIPTOR_TYPE_DEVICE, 0, reinterpret_cast<uint8_t*>(&desc), sizeof(StandardDeviceDescriptor));
    if (rc != USB_OK) {
        return false;
    }
    if (desc.bDeviceClass == 9) { // hub?
        return true;
    }
    return false;
}

int UsbHub::SetPortPower(int port)
{
    return SetPortFeature(PORT_POWER, port);
}

int UsbHub::ClearPortPower(int port)
{
    return ClearPortFeature(PORT_POWER, port);
}

int UsbHub::SetPortFeature(int feature, int index)
{
    return m_ctlEp->controlSend(0x23, SET_FEATURE,feature,index,0,0);
}

int UsbHub::ClearPortFeature(int feature, int index)
{
    return m_ctlEp->controlSend(0x23, CLEAR_FEATURE,feature,index,0,0);
}

int UsbHub::SetPortReset(int port)
{
    return SetPortFeature(PORT_RESET, port);
}

int UsbHub::GetPortStatus(int port, uint32_t* status)
{
    return m_ctlEp->controlReceive(0xa3, GET_STATUS, 0, port, (uint8_t*)status, 4);
}

int UsbHub::PortReset(int port)
{
    DBG("%p port=%d\n", this, port);
    TEST_ASSERT(port >= 1);
    SetPortReset(port);
    // wait reset
    for(int i = 0; i < 100; i++) {
        uint32_t status;    
        GetPortStatus(port, &status);
        DBG("RESET port: %d status: %08X\n", port, status);
        if (status & 0x100000) { // Reset change , Reset complete
            return USB_OK;
        }
        wait_ms(5);
     }
     return USB_ERROR;
}
