// AD-12864-SPI test program
// About AD-12864-SPI, see http://www.aitendo.co.jp/product/1622.

// Pin allocation
// 1 p21 #CS1 with 10k ohm pull-up
// 2 p22 #RESET with 10k ohm pull-up
// 3 p23 A0 ... 0:command 1:data
// 4 p13 SCK
// 5 p11 MOSI
// 6     Vdd
// 7     Vss
// 8 NC  LED_A

// Japanese Character Version:
// UTF Font from e-font http://openlab.ring.gr.jp/efont/unicode/
// and row-col convert with information from:
// http://todotani.cocolog-nifty.com/blog/2009/07/arduino-d98c.html
// Font data file based on
// http://todotani.cocolog-nifty.com/blog/files/glcdfont_12_ucs2.zip
// make binary and split binary 

#include "mbed.h"

DigitalOut cs(p21);
DigitalOut rst(p22);
DigitalOut a0(p23);
SPI spi(p11, p12, p13); // mosi, miso, sclk

LocalFileSystem local("local");

#ifdef DEBUG
Serial pc(USBTX, USBRX); // tx, rx
#endif

void regwrite(unsigned char c) {
    cs = a0 = 0;
    spi.write(c);
    cs = 1;
}

void datawrite(unsigned char c) {
    cs = 0;
    a0 = 1;
    spi.write(c);
    cs = 1;
}

// set position (x, 8*y)
void locate(int x, int y) {
    regwrite(0xb0 | (y & 0x0f)); // Page Address Set (see 2.4.3)
    regwrite(0x10 | (x >> 4 & 0x0f)); // Column Address Set (see 2.4.4)
    regwrite(x & 0x0f);
}

void cls(void) {
    int x, y;
    for (y = 0; y < 8; y++) {
        locate(0, y);
        for (x = 0; x < 128; x++) datawrite(0x00);
    }
}

void plot(int x, int y) {
    locate(x, y >> 3);
    datawrite(1 << (y & 7));
}

void init() {
    spi.format(8,0); // nazo
    spi.frequency(10000000); // modify later

    // reset
    wait_ms(200);
    rst = 0;
    wait_ms(200);
    rst = 1;

    // initialize sequence
    regwrite(0xaf);    // display on (see 2.4.1)
    regwrite(0x2f);    // power control set (see 2.4.16)
    regwrite(0x81);    // set electronic volume mode (see 2.4.18)
//    regwrite(0x1f);    // electronic volume data 00-3f
    regwrite(0x00);    // electronic volume data 00-3f
    regwrite(0x27);    // V5 Volatge Regulator Internal Resister Ratio Set (see 2.4.17)
    regwrite(0xa2);    // LCD Bias Set ... 1/9 bias (see 2.4.11)
    regwrite(0xc8);    // Common Output Mode Select ... Reverse (see 2.4.15)
    regwrite(0xa0);    // ADC Select ... Normal (see 2.4.8)
    regwrite(0xa4);    // Display All Points ON/OFF ... normal (see 2.4.10)
    regwrite(0xa6);    // Display Normal/Reverse ... normal (see 2.4.9)
    regwrite(0xac);    // Static Indicator ... off (see 2.4.19)
    regwrite(0x00);    // off
    regwrite(0x40);    // Display Strat Line Set ... 0 (see 2.4.2)
    regwrite(0xe0);    // Write Mode Set
}

void drawchar24(int x, int y, unsigned int c) {
    unsigned char buf[24];
    char fname[20];
    int width,base;

    if((c >= 0x0020) && (c <= 0x00FE)) {
      width=12;
      base=0x0020;
      sprintf(fname,"/local/002000FE.dat");
    }
    else if((c >= 0x0100) && (c <= 0x05F3)) {
      width=12;
      base=0x0100;
      sprintf(fname,"/local/010005F3.dat");
    }
    else if((c >= 0x1E00) && (c <= 0x1F72)) {
      width=12;
      base=0x1E00;
      sprintf(fname,"/local/1E001F72.dat");
    }
    else if((c >= 0x2010) && (c <= 0x28FE)) {
      width=12;
      base=0x2010;
      sprintf(fname,"/local/201028FE.dat");
    }
    else if((c >= 0x3000) && (c <= 0x33DD)) {
      width=24;
      base=0x3000;
      sprintf(fname,"/local/300033DD.dat");
    }
    else if((c >= 0x4E00) && (c <= 0x9FA4)) {
      width=24;
      base=0x4E00;
      sprintf(fname,"/local/4E009FA4.dat");
    }
    else if((c >= 0xF900) && (c <= 0xFA26)) {
      width=24;
      base=0xF900;
      sprintf(fname,"/local/F900FA26.dat");
    }
    else if((c >= 0xFF00) && (c <= 0xFF5E)) {
      width=24;
      base=0xFF00;
      sprintf(fname,"/local/FF00FF5E.dat");
    }
    else if((c >= 0xFF60) && (c <= 0xFF9F)) {
      width=12;
      base=0xFF60;
      sprintf(fname,"/local/FF60FF9F.dat");
    }
    else if((c >= 0xFFE0) && (c <= 0xFFE6)) {
      width=24;
      base=0xFFE0;
      sprintf(fname,"/local/FFE0FFE6.dat");
    }
    else {
      return;
    }
    
    FILE *fp = fopen(fname,"rb");
    fseek(fp,(c-base) * width,SEEK_SET);
    fread(buf, sizeof(unsigned char), width, fp);
    fclose(fp);

    #ifdef DEBUG
    int i;
    pc.printf("DEBUG:%s:%04.4x -> ",fname,c);
    for(i=0;i<width;i++) {
        pc.printf("%02.2x ",buf[i]);
    }
    pc.printf("\r\n");
    #endif
        
    if(width == 24) {
      locate(x,y);
      datawrite(buf[0]); datawrite(buf[1]); datawrite(buf[2]);
      datawrite(buf[3]); datawrite(buf[4]); datawrite(buf[5]);
      datawrite(buf[6]); datawrite(buf[7]); datawrite(buf[8]);
      datawrite(buf[9]); datawrite(buf[10]); datawrite(buf[11]);
      locate(x,y+1);
      datawrite(buf[12]>>4); datawrite(buf[13]>>4); datawrite(buf[14]>>4);
      datawrite(buf[15]>>4); datawrite(buf[16]>>4); datawrite(buf[17]>>4);
      datawrite(buf[18]>>4); datawrite(buf[19]>>4); datawrite(buf[20]>>4);
      datawrite(buf[21]>>4); datawrite(buf[22]>>4); datawrite(buf[23]>>4);
    }
    else if(width == 12) {
      locate(x,y);
      datawrite(buf[0]); datawrite(buf[1]); datawrite(buf[2]);
      datawrite(buf[3]); datawrite(buf[4]); datawrite(buf[5]);
      locate(x,y+1);
      datawrite(buf[6]); datawrite(buf[7]); datawrite(buf[8]);
      datawrite(buf[9]); datawrite(buf[10]); datawrite(buf[11]);
    }
}
/*
void drawtext(const char *s) {
    unsigned char c;
    while ((c = *s++) != '\0') drawchar(c);
}
*/

int main() {
    init();
    cls();
    locate(0, 0);
    // http://www.unicode.org/charts/PDF/U3040.pdf
    // http://ash.jp/code/unitbl21.htm
    drawchar24(0,0,0x3053);  // KO on UCS2
    drawchar24(12,0,0x3093); // N
    drawchar24(24,0,0x306b); // NI
    drawchar24(36,0,0x3061); // CHI
    drawchar24(48,0,0x306F); // HA
    drawchar24(0,2,0x65E5);  // NI
    drawchar24(12,2,0x672C); // HON
    drawchar24(24,2,0x8A9E); // GO
    drawchar24(36,2,0x8868); // HYO
    drawchar24(48,2,0x793A); // JI
    drawchar24(60,2,0x306E); // NO
    drawchar24(72,2,0x30C6); // TE
    drawchar24(84,2,0x30B9); // SU
    drawchar24(96,2,0x30C8); // TO
    drawchar24(108,2,0x3002);// MARU
    drawchar24(0,4,0xFF21);   // zenkaku-A
    drawchar24(12,4,0xFF22);  // zenkaku-B
    drawchar24(24,4,0xFF23);  // zenkaku-C
    drawchar24(36,4,0xFF24);  // zenkaku-D
    drawchar24(48,4,0xFF25);  // zenkaku-E
    drawchar24(60,4,0xFF26);  // zenkaku-F
    drawchar24(72,4,0xFF27);  // zenkaku-G
    drawchar24(84,4,0xFF28);  // zenkaku-H
    drawchar24(96,4,0xFF29);  // zenkaku-I
    drawchar24(108,4,0xFF2A); // zenkaku-J

    while (1) {}
}
