/* import this librairies : 
*       TextLCD.h
* online compiler : https://mbed.org/compiler/#import:https://mbed.org/users/simon/code/TextLCD/;mode:lib
* file.zip :        http://mbed.org/users/simon/code/TextLCD/archive/44f34c09bd37.zip
*
*       Led.h
* online compiler : https://mbed.org/compiler/#import:https://mbed.org/users/us191/code/LED/;mode:lib
* file.zip :        https://mbed.org/users/us191/code/LED/archive/75240e7c857c.zip
*/

#ifndef Display_H
#define Display_H

#include "mbed.h"
#include "TextLCD.h"
#include "Led.h"
#include "string"
#include <vector>


/** This class manage 3 LED and 1 TextLCD display.
 *  She permit to print messages and to light or flash LED.
 *  
 * @code
 * #include "mbed.h"
 * #include "Display.h"
 *
 * Display disp(p12, p13, p14, p15, p16, p5, p6, p7, p8); // greenLED, redLED, orangeLED, LCD : rs, e, d4-d7 (default : 16X2 screen)
 *
 * float delay = 3;
 *
 *
 * int main() {
 *
 *   // print "HelloWorld !" on LCD screen and put on greenLED, flash redLED and put off orangeLED 
 *   disp.printMessage("Hello World !", Display::on, Display::flash, Display::off);
 *   wait(delay);
 *   
 *   // print "coucou !" on LCD screen and put off greenLED, put on redLED and flash orangeLED
 *   disp.printMessage("coucou !", Display::off, Display::on, Display::flash);
 *   wait(delay);
 *   
 *   // print "bye bye !" on LCD screen and flash greenLED, put off redLED and put on orangeLED
 *   disp.printMessage("bye bye !", Display::flash, Display::off, Display::on);
 *   wait(delay);
 *
 *   // clear LCD screen and shutdown 3 LED
 *   disp.clear();
 * }
 * @endcode
 */

class Display {
   
public :

    /** use by printMessage() and changeStatusLed() */    
    enum choiceStatusLED {
        on      /**< put on led */
        , off   /**< put off led */
        , flash /**< flash led */
    };
    

    /** Create a Display interface (default : 16X2 screen)
    * 
    * @param pinGreenLED    greenLED broche
    * @param pinRedLED      redLED broche
    * @param pinOrangeLED   orangeLED broche
    * @param pinLCDrs       LCD Instruction/data control line
    * @param pinLCDe        LCD Enable line (clock)
    * @param pinLCDd0-d3    LCD Data lines     
    */
    Display(PinName pinGreenLED, PinName pinRedLED, PinName pinOrangeLED, PinName pinLCDrs, PinName pinLCDe,
            PinName pinLCDd0, PinName pinLCDd1, PinName pinLCDd2, PinName pinLCDd3);

    /** Create a Display interface
    * 
    * @param pinGreenLED    greenLED broche
    * @param pinRedLED      redLED broche
    * @param pinOrangeLED   orangeLED broche
    * @param pinLCDrs       LCD Instruction/data control line
    * @param pinLCDe        LCD Enable line (clock)
    * @param pinLCDd0-d3    LCD Data lines
    * @param typeScreen     LCD Sets the panel size/addressing mode (default = LCD16x2)
    */
    Display(PinName pinGreenLED, PinName pinRedLED, PinName pinOrangeLED, PinName pinLCDrs, PinName pinLCDe,
            PinName pinLCDd0, PinName pinLCDd1, PinName pinLCDd2, PinName pinLCDd3, TextLCD::LCDType typeScreen);
    
    /** Destructor
    */
    ~Display(void);

      
    /** print message on LCD screen and change status of green, red and orange LED.
    * all line except the last line, have to finish by CRLF. Warning : CRLF count like a character.
    * don't end message by CRLF.
    * examples :\n
    * "this line have to finish by CRLF\/n
    * and next too\/n
    * but not the last"
    *
    * @param message            message will be display on screen
    * @param statusGreenLED     new status of greenLED (on, off, flash)
    * @param statusRedLED       new status of redLED (on, off, flash)
    * @param statusOrangeLED    new status of orangeLED (on, off, flash)
    */
    void printMessage(string message, choiceStatusLED statusGreenLED, choiceStatusLED statusRedLED, choiceStatusLED statusOrangeLED);
    
    /** put off 3 LED and clear LCD screen
    */
    void clear(void);

    
protected :

    /* 3 led manage by current object */
    Led _greenLED, _redLED, _orangeLED;
    /* LCD screen manage by current object */
    TextLCD _lcd;
    
    /* use by changeStatusLed() */
    enum choiceLED {
        green       /*< choice greenLED */
        , red       /*< choice redLED */
        , orange    /*< choice orangeLED */
    };
    
    
    
    /* change led status to statusLED
    * use by printMessage()
    *
    * @param led        the led (green, red, orange)
    * @param statusLED  new status (on, off, flash)
    */
    void changeStatusLED(choiceLED led, choiceStatusLED statusLED);
    
    /* put off 3 LED
    * use by clear()
    */
    void shutdownLED(void);

    /* clear LCD screen
    * use by clear()
    */
    void cls(void);
    
    /* check if this message can be display correctly on LCD screen 
    * use by printMessage()
    *
    * @param message    the message to be verified
    * @return 
    *        true if message is ok
    *        flase else
    */
    bool checkMessage(string message);
    
    /* calcul number of '\n'
    * use by checkMessage()
    *
    * @param message    the message to be analyse
    * @return number of '\n'
    */    
    int calculNbCRLF(string message);
    
    /* return vector of all line message
    * use by checkMessage()
    *
    * @param message    the message to be cut
    * @return   a vector which contains a line of the message by entry
    */
    vector<string> subAllLine(string message);
    
};

#endif // Display_H