/*
 * MCP4822A DAC array library demonstration program.
 *
 * Copyright (c) 2011 Steven Beard, UK Astronomy Technology Centre.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"
#include "MCP4822A.h"

/*
 * In this example there are 4 DAC chips daisy-chained on the SPI bus,
 * which together provide 8 channels.
 */
const int ndacs = 4;
const int nchans = ndacs * 2;

/*
 * The following mbed pins are used:
 *
 * p11 - SPI mosi
 * p13 - SPI sclk
 * p14, p15, p16, p17 - CS pins, each connected to a separate DAC.
 * p21 - LDAC pin, connected to all DAC chips.
 */
PinName cspins[] = {(p14),(p15),(p16),(p17)};
MCP4822A MCP(ndacs, p11, p13, cspins, p21);    // MOSI, SCLK, nCS-list, nLdac

int main() {
    int i, buffer;

    printf("\r\n\r\nSetting SPI clock frequency to 10 MHz.\r\n");
    MCP.frequency(10000000);
    
    // Begin by setting all the DACs to 1000 millivolts at gain 2.
    printf("Loading all channels with 1000mV at gain 2...\r\n");
    MCP.latch_disable();
    for (i=0; i<ndacs; i++) {
        MCP.writeA2(i, 1000);
        MCP.writeB2(i, 1000);
    }
    wait(4);
    printf("Latching now.\n");
    MCP.latch_enable();
    wait(4);

    // Initialise an array of test voltages (in mV).
    int test_voltages[nchans];
    for (i=0; i<nchans; i++) {
        test_voltages[i] = i * 500;
    }

    while (1) {

        // Write the array of test voltages to the DACs
        // (at gain 2 with automatic latching).
        printf("Writing: ");
        for (i=0; i<nchans; i++) {
            printf("%d ", test_voltages[i]);
        }
        printf("\r\n");
        MCP.write(nchans, test_voltages, 2, 1);
        wait(4);

        //Move the values in the array along one place.
        buffer = test_voltages[0];
        for (i=0; i<(nchans-1); i++) {
            test_voltages[i] = test_voltages[i+1];
        }
        test_voltages[nchans-1] = buffer;
    }
}