//copyright 2011 Uehara Yoshiyuki
//====================================================================
//The author provide the programs without any guarantees or warranty.
//The author is not responsible for any damage or losses of any kind 
//caused by using or misusing of the programs.
//The author is under no obligation to provide support, service, 
//corrections, or upgrades to the programs.
//====================================================================
// MAPLE board[MARM01-BASE]
// operation console utilizing text-LCD and buttons
#include "Maple_console.h"
#include "Maple_alarm_clock.h"
#include "Maple_test.h"
#include "Maple_RTC.h"
#include "Maple_I2C.h"
#include "Maple_LCD.h"
#include "Maple.h"
#include "mbed.h"

// global variables
int console_mode;       // console function selected
int console_cursor;     // cursor position
int console_display;    // console display on/off
int button_count[BUTTON_SIZE];
int button_status[BUTTON_SIZE];
int test_mode;          // used within test

// timer interrupt to refresh console
Ticker console_refresh_tick;

// initialize console
void console_initialize() {
    enter_mode_clock();
    console_display = DISPLAY_ON;
    for (int i = 0; i < BUTTON_SIZE; ++i) {
        button_count[i] = 0;
        button_status[i] = BUTTON_IDLE;
    }
    console_refresh_tick.attach(&console_refresh, CONSOLE_REFRESH_RATE);
}

// refresh console called by ticker interrupt
void console_refresh() {
    refresh_display();
    button_action();
}

// read RTC and print to LCD
static void refresh_display() {
    char row0[17], row1[17];
    int cursor_r, cursor_c;

    switch(console_mode) {
    case MODE_ADJUST:  display_adjust(row0, row1, cursor_r, cursor_c);  break;
    case MODE_ALARM:   display_alarm( row0, row1, cursor_r, cursor_c);  break;
    case MODE_TIMER:   display_timer( row0, row1, cursor_r, cursor_c);  break;
    case MODE_TEST:    display_test(  row0, row1, cursor_r, cursor_c);  break;
    default:           display_clock( row0, row1, cursor_r, cursor_c);
    }
    if(console_display == DISPLAY_ON) {
        LCD_cursor_off();
        LCD_locate(0, 0);  LCD_print_string(row0);
        LCD_locate(1, 0);  LCD_print_string(row1);
        LCD_locate(cursor_r, cursor_c);
        LCD_cursor_on();
    }
}

// read button and launch action
//   called by timer tick interrupt
static void button_action() {
    char r = i2c_BTN_read();
    button_check(r, BUTTON_A    );
    button_check(r, BUTTON_B    );
    button_check(r, BUTTON_LEFT );
    button_check(r, BUTTON_DOWN );
    button_check(r, BUTTON_RIGHT);
    button_check(r, BUTTON_UP   );
    if(button_trigger(BUTTON_A,     0, 0)) { button_exit();         }
    if(button_trigger(BUTTON_B,     1, 0)) { button_function();     }
    if(button_trigger(BUTTON_LEFT,  1, 0)) { button_cursor_move(0); }
    if(button_trigger(BUTTON_RIGHT, 1, 0)) { button_cursor_move(1); }
    if(button_trigger(BUTTON_DOWN,  1, 1)) { button_xxcrement(0);   }
    if(button_trigger(BUTTON_UP,    1, 1)) { button_xxcrement(1);   }
}

// button check and count up
static void button_check(int r, int button_number) {
    if((r & (0x01 << button_number)) == BUTTON_ON) {
        ++button_count[button_number];
    }
    else {
        button_count[button_number] = 0;
        button_status[button_number] = BUTTON_IDLE;
    }
}

// button action trigger
//   b: button number
//   r: repeat enable
//   f: fast-repeat enable
static bool button_trigger(int b, int r, int f) {
    if((button_status[b] == BUTTON_IDLE) && (button_count[b] > BUTTON_THRESHOLD)) {
        button_status[b] = BUTTON_BUSY;
        return true;    // one shot
    }
    else if((f != 0) && (button_count[b] > BUTTON_FAST) && (button_count[b] % BUTTON_PERIOD_FAST == 0)) {
        return true;    // repeat fast
    }
    else if((r != 0) && (button_count[b] > BUTTON_REPEAT) && (button_count[b] % BUTTON_PERIOD_REPEAT == 0)) {
        return true;    // repeat
    }
    else {
        return false;
    }
}

// exit to normal mode
void button_exit() {
    LCD_clear_display();
    console_display = DISPLAY_ON;
    enter_mode_clock();
}

// select a function
static void button_function() {
    switch(console_mode) {
    case MODE_CLOCK:   enter_mode_adjust();  break;
    case MODE_ADJUST:  enter_mode_alarm();   break;
    case MODE_ALARM:   enter_mode_timer();   break;
    case MODE_TIMER:   enter_mode_test();    break;
    case MODE_TEST:    test_function();      break;
    default:           button_exit();
    }
}

// move cursor
//   flag  0:left, 1:right
static void button_cursor_move(int flag) {
    switch(console_mode) {
    case MODE_ADJUST:  cursor_move_adjust(flag);  break;
    case MODE_ALARM:   cursor_move_alarm( flag);  break;
    case MODE_TIMER:   cursor_move_timer( flag);  break;
    case MODE_TEST:    cursor_move_test(  flag);  break;
    default:           cursor_move_clock( flag);
    }
}

// move cursor common
//   flag  0:left, 1:right
void cursor_move(int flag, int cursor_size) {
    switch(flag) {
    case 0:     // move left .. decrement
        if(console_cursor > 0) {
            --console_cursor;
        }
        else {
            console_cursor = cursor_size - 1;
        }
        break;
    default:    // move right .. increment
        ++console_cursor;
        if(console_cursor >= cursor_size) {
            console_cursor = 0;
        }
    }
}

// increment/decrement
//   flag 0:decrement, 1:increment
static void button_xxcrement(int flag) {
    switch(console_mode) {
    case MODE_ADJUST:
        button_xxcrement_adjust(flag);
        break;
    case MODE_ALARM:
        button_xxcrement_alarm(flag);
        break;
    case MODE_TIMER:
        button_xxcrement_timer(flag);
        break;
    case MODE_TEST:
        button_xxcrement_test(flag);
        break;
    default:
        button_xxcrement_clock(flag);
    }
}
