//copyright 2011 Uehara Yoshiyuki
//====================================================================
//The author provide the programs without any guarantees or warranty.
//The author is not responsible for any damage or losses of any kind 
//caused by using or misusing of the programs.
//The author is under no obligation to provide support, service, 
//corrections, or upgrades to the programs.
//====================================================================
// MAPLE board[MARM01-BASE]
// LCD(TC1602E-25A) driver
// - 4-bit I/O port.
// - "r/w" is fixed to "w". Read instructions do not work.
#include "Maple_LCD.h"
#include "Maple.h"
#include "mbed.h"

// LCD interface
DigitalOut LCD_rs(p25);
DigitalOut LCD_e(p24);
BusInOut   LCD_d4_d7(p12, p13, p14, p23);
int        LCD_rw;  // dummy .. fixed to 0 on MAPLE Board

// functions for timing
static void LCD_bus_switch_time() {
    __nop();
}

static void LCD_data_hold_time() {
    __nop();
}

static void LCD_data_setup_time() {
    wait_us(1);
}

static void LCD_execution_time() {
    wait_us(60);
}

static void LCD_clear_time() {
    wait_us(2500);
}

// rw    1 1 0 0 0 0 0 0 0 1 1 1 1 1
// data  - - - H H H H H - - - - - - 
// e     0 0 0 1 1 1 1 0 0 0 0 0 0 0
//             --1us--     ---60us---
static void LCD_write_1(int rs, char w) {
    LCD_rs = rs;
    LCD_rw = 0;
    LCD_bus_switch_time();
    LCD_d4_d7.output();

    LCD_e = 1;
    LCD_d4_d7 = w >> 4;
    LCD_data_setup_time();
    LCD_e = 0;

    LCD_d4_d7.input();
    LCD_bus_switch_time();
    LCD_rw = 1;

    LCD_execution_time();
}

// rw    1 1 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 
// data  - - - H H H H H L L L L L - - - - - - 
// e     0 0 0 1 1 1 1 0 1 1 1 1 0 0 0 0 0 0 0 
//             --1us--   --1us--     ---60us---
static void LCD_write_2(int rs, char w) {
    LCD_rs = rs;
    LCD_rw = 0;
    LCD_bus_switch_time();
    LCD_d4_d7.output();

    LCD_e = 1;
    LCD_d4_d7 = w >> 4;
    LCD_data_setup_time();
    LCD_e = 0;
    LCD_data_hold_time();

    LCD_e = 1;
    LCD_d4_d7 = w;
    LCD_data_setup_time();
    LCD_e = 0;

    LCD_d4_d7.input();
    LCD_bus_switch_time();
    LCD_rw = 1;

    LCD_execution_time();
}

// rw    1 1 1 1 1 1 1 1 1 1 1 1 1
// data  - - - H H H H - L L L L - 
// e     0 0 1 1 1 1 0 1 1 1 1 0 0 
//             setup     setup     
static unsigned char LCD_read_2(int rs) {
    char r;

    LCD_rs = rs;
    LCD_e = 1;
    LCD_data_setup_time();
    r = LCD_d4_d7 << 4;
    LCD_e = 0;
    LCD_data_hold_time();
 
    LCD_e = 1;
    LCD_data_setup_time();
    r += LCD_d4_d7;
    LCD_e = 0;
    LCD_data_hold_time();

    return r;
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  0  0  0  0  0  1 :0x01
void LCD_clear_display() {
    LCD_write_2(0, 0x01);
    LCD_clear_time();
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  0  0  0  0  1  - :0x02
void LCD_return_home() {
    LCD_write_2(0, 0x02);
    LCD_clear_time();
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  0  0  0  1  id s :0x04
//  id: curdor move direction.. 0: decrement, 1: increment 
//  s: accompanies display shift.. 0: no, 1: yes 
void LCD_entry_mode_set(int id, int s) {
    char w;

    w = 0x04;
    if(id != 0) { w += 0x02; }
    if(s  != 0) { w += 0x01; }
    LCD_write_2(0, w);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  0  0  1  d  c  b :0x08
//  d: display.. 0: off, 1: on 
//  c: cursor.. 0: off, 1: on 
//  b: blinking.. 0: off, 1: on
void LCD_display_on_off_control(int d, int c, int b) {
    char w;

    w = 0x08;
    if(d != 0) { w += 0x04; }
    if(c != 0) { w += 0x02; }
    if(b != 0) { w += 0x01; }
    LCD_write_2(0, w);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  0  1  sc rl -  - :0x10
//  sc: shift/cursor.. 0: cursor move, 1: display shift
//  rl: right/left.. 0: shift to the left, 1: shift to the right
void LCD_cursor_or_display_shift(int sc, int rl) {
    char w;

    w = 0x10;
    if(sc != 0) { w += 0x08; }
    if(rl != 0) { w += 0x04; }
    LCD_write_2(0, w);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  0  1  dl n  f  -  - :0x20
//
// "function set" is used twice at initialization only
// 1st use: 8 bit write
//   dl: data length.. 0: 4 bits, 1: 8 bits --> 0 fixed
//
// 2nd use: 4 bit write
//   dl = 0 (must not change) 
//   n: line select.. 0: 1 line, 1: 2 lines 
//   f: font select.. 0: 5*8 dots, 1: 5*10 dots
void LCD_function_set(int n, int f) {
    char w;

    w = 0x20;
    if(n != 0) { w += 0x08; }
    if(f != 0) { w += 0x04; }
    LCD_write_1(0, w);
    LCD_write_2(0, w);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 0  1  a5 a4 a3 a2 a1 a0:0x40
//  a5-a0: CGRAM address
void LCD_set_CGRAM_address(int a) {
    LCD_write_2(0, 0x40 + (a & 0x3f));
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  0 : 1  a6 a5 a4 a3 a2 a1 a0:0x80
//  a6-a0: DDRAM address
void LCD_set_DDRAM_address(int a) {
    LCD_write_2(0, 0x80 + (a & 0x7f));
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 0  1 : bf a6 a5 a4 a3 a2 a1 a0
//  bf: busy flag- 0: not busy, 1: busy 
//  a6-a0: DDRAM address
char LCD_read_busy_flag_and_address() {
    return LCD_read_2(0);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 1  0 : d7 d6 d5 d4 d3 d2 d1 d0
//  d7-d0: DDRAM address
void LCD_write_data_to_CG_or_DDRAM(char d) {
    LCD_write_2(1, d);
}

// rs rw: d7 d6 d5 d4 d3 d2 d1 d0
// 1  1 : d7 d6 d5 d4 d3 d2 d1 d0
//  d7-d0: read data from CGRAM or DDRAM
char LCD_read_data_from_CG_or_DDRAM() {
    return LCD_read_2(1);
}

// an example of CGRAM customizing
void LCD_CGRAM_customize() {
    LCD_set_CGRAM_address(0x00);
                                            // 0x00
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x01
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x1f);    // 11111
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x02
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x03
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x04
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x05
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x1f);    // 11111
    LCD_write_data_to_CG_or_DDRAM(0x15);    // 1-1-1
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x06
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0a);    // -1-1-
    LCD_write_data_to_CG_or_DDRAM(0x11);    // 1---1
    LCD_write_data_to_CG_or_DDRAM(0x1f);    // 11111
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x0e);    // -111-
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
                                            // 0x07
    LCD_write_data_to_CG_or_DDRAM(0x04);    // --1--
    LCD_write_data_to_CG_or_DDRAM(0x06);    // --11-
    LCD_write_data_to_CG_or_DDRAM(0x05);    // --1-1
    LCD_write_data_to_CG_or_DDRAM(0x05);    // --1-1
    LCD_write_data_to_CG_or_DDRAM(0x0d);    // -11-1
    LCD_write_data_to_CG_or_DDRAM(0x1c);    // 111--
    LCD_write_data_to_CG_or_DDRAM(0x18);    // 11---
    LCD_write_data_to_CG_or_DDRAM(0x00);    // -----
}

// an example of initializing LCD module .. standard use
void LCD_initialize() {
    LCD_function_set(1, 0);                  // 2 lines, 5*8dots font set
    LCD_display_on_off_control(1, 0, 0);     // display on, cursor off, blinking off
    LCD_entry_mode_set(1, 0);                // increment yes, shift no
    LCD_CGRAM_customize();                   // CGRAM customize by an example above
    LCD_clear_display();                     // select 1st row and clear display 
}

// print char to LCD module
void LCD_print_char(char c) {
    LCD_write_data_to_CG_or_DDRAM(c);
}

// print string to LCD module
void LCD_print_string(char s[]) {
    for(int i = 0; s[i] != '\0'; ++i) {
        LCD_write_data_to_CG_or_DDRAM(s[i]);
    }
}

// print integer(1 byte) in hex format to LCD module
void LCD_print_hex(int i) {
    LCD_write_data_to_CG_or_DDRAM(int_to_hex1(i >> 4));
    LCD_write_data_to_CG_or_DDRAM(int_to_hex1(i));
}

// locate LCD module
//   row    = 0 .. 1
//   column = 0 .. 0x3f
void LCD_locate(int row, int column) {
    LCD_set_DDRAM_address(row * 0x40 + column);
}

// set cursor on, blink off
void LCD_cursor_on() {
    LCD_display_on_off_control(1, 1, 0);
}

// set cursor off, blink off
void LCD_cursor_off() {
    LCD_display_on_off_control(1, 0, 0);
}
