#ifndef SN_SnStatusFrame
#define SN_SnStatusFrame

#include <stdint.h>

#include "SnBitUtils.h"
#include "SnCommWin.h"
#include "SnConfigFrame.h"
#include "SnEventFrame.h"
#include "SnTempFrame.h"
#include "SnSDUtils.h"
#include "Watchdog.h"

//#define ST_DEBUG

struct SnStatusFrame {
    //
    // A simple struct (everything public) to function like namespace.
    // The contents of the status frame are sent from here, in order to
    // help make sure the status frame is the same for each comm method.
    //
    // No member variables are used in order to preserve memory on the mbed.
    // (i.e. no actual status frame middle-man object exists)
    //
    
    static const uint8_t    kIOVers;    // MUST BE INCREASED if bytes written/read change!!
    
    static const uint32_t   kMaxSizeOfV1 =
               1u + sizeof(uint64_t)
            + (sizeof(uint32_t)*3u) + (sizeof(uint16_t))
            + (sizeof(uint8_t)*3u) + SnConfigFrame::kConfLblLen
            + SnEventFrame::kMaxSizeOfV1;
    static const uint32_t   kMaxSizeOfV2 = 
            sizeof(uint64_t) + (3u*sizeof(uint32_t)) + (2u*sizeof(uint16_t))
            + (3u*sizeof(uint8_t)) + (2u*sizeof(float))
            + SnConfigFrame::kConfLblLen;
    static const uint32_t   kMaxSizeOfV3 = 
            kMaxSizeOfV2 + sizeof(uint32_t) + sizeof(float);
    static const uint32_t   kMaxSizeOfV4 = kMaxSizeOfV3 - sizeof(uint32_t);
    static const uint32_t   kMaxSizeOfV5 = kMaxSizeOfV4 + sizeof(uint8_t);
    static const uint32_t   kMaxSizeOfV6 = kMaxSizeOfV5 + sizeof(uint8_t) + sizeof(uint32_t); // iov and power on time
    static const uint32_t   kMaxSizeOfV7 = kMaxSizeOfV6 + sizeof(float); // temperature
    static const uint32_t   kMaxSizeOfV8 = kMaxSizeOfV7 + sizeof(uint8_t);
    static const uint32_t   kMaxSizeOfV9 = kMaxSizeOfV8; // V9 just marks that no event is automatically sent along
    static const uint32_t   kMaxSizeOfV10 = kMaxSizeOfV9
            - (2u*sizeof(float))    // remove thmrate, evtrate
            + (2u*sizeof(uint32_t)) // add num thm trigs, num saved evts
            + sizeof(float);        // add seq livetime
    static const uint32_t   kMaxSizeOf   = kMaxSizeOfV1;
    

 public:
    static bool             fgRecalcFiles;
 private:
    static float            fgFreeMB;

 public:
    static
    float       GetFreeMB() { return fgFreeMB; }
    
    static
    uint32_t    GetMaxSizeOf(const uint8_t rv) {
        if (rv==1) {
            return kMaxSizeOfV1;
         } else if (rv==2) {
            return kMaxSizeOfV2;
         } else if (rv==3) {
            return kMaxSizeOfV3;
         } else if (rv==4) {
            return kMaxSizeOfV4;
         } else if (rv==5) {
            return kMaxSizeOfV5;
         } else if (rv==6) {
            return kMaxSizeOfV6;
         } else if (rv==7) {
            return kMaxSizeOfV7;
         } else if (rv==8) {
            return kMaxSizeOfV8;
         } else if (rv==9) {
            return kMaxSizeOfV9;
         } else if (rv==10) {
            return kMaxSizeOfV10;
         } else {
            return kMaxSizeOf;
         }
    }
    
    template<class T>
    static
    SnCommWin::ECommWinResult WriteTo(T& x,
                                      const SnConfigFrame& conf,
                                      const uint16_t seq,
                                      const uint32_t numThmTrigs,
                                      const uint32_t numSavedEvts, 
                                      const float    seqlive,
                                      const uint32_t powerOnTime,
                                      const SnTempFrame& temper,
                                      const uint8_t loseLSB,
                                      const uint8_t loseMSB,
                                      const uint16_t wvBase) {
        // expect 'x' to be a MODSERIAL& or a char const*
        
        const uint32_t llen = strlen(conf.GetLabel());
        
        // if anything about these writes changes, update kIOVers and SizeOf
        x = SnBitUtils::WriteTo(x, SnStatusFrame::kIOVers);
        x = SnBitUtils::WriteTo(x, conf.GetMacAddress());
        x = SnBitUtils::WriteTo(x, llen);
        x = SnBitUtils::WriteTo(x, conf.GetLabel(), llen);
        x = SnBitUtils::WriteTo(x, conf.GetRun());
        x = SnBitUtils::WriteTo(x, seq);
        x = SnBitUtils::WriteTo(x, static_cast<uint32_t>(time(0)));
        x = SnBitUtils::WriteTo(x, loseLSB);
        x = SnBitUtils::WriteTo(x, loseMSB);
        x = SnBitUtils::WriteTo(x, wvBase);
//        x = SnBitUtils::WriteTo(x, thmrate);
//        x = SnBitUtils::WriteTo(x, evtrate);
        // file info
        if (fgRecalcFiles) {
            /*
#ifdef ST_DEBUG
            printf("Calling GetDirProps for %s\r\n",SnSDUtils::kSDdir);
#endif
            SnSDUtils::GetDirProps(SnSDUtils::kSDdir, fgNfiles, fgTotKbytes);
            fgTotKbytes /= 1e3; // KB
#ifdef ST_DEBUG
            printf("nfiles=%u, tb=%g kb\r\n",fgNfiles,fgTotKbytes);
#endif
            */
            fgFreeMB = SnSDUtils::GetFreeBytes() / 1048576.0;
#ifdef ST_DEBUG
            printf("fgFreeMB = %g\r\n",fgFreeMB);
#endif
            fgRecalcFiles = false;
        }
        x = SnBitUtils::WriteTo(x, fgFreeMB);
        
        // watchdog reset bit cleared after status received successfully, not here
        const uint8_t wdreset = Watchdog::didWatchdogReset();
        x = SnBitUtils::WriteTo(x, wdreset);
        
        // the config i/o version we expect (might be needed to send a new config to the station)
        x = SnBitUtils::WriteTo(x, conf.kIOVers);

        x = SnBitUtils::WriteTo(x, powerOnTime);
        x = SnBitUtils::WriteTo(x, temper.GetTemperature());
        x = SnBitUtils::WriteTo(x, static_cast<uint8_t>(SnSDUtils::IsInitOk()));
        
        // send number of thm trigs, num saved events and livetime
        x = SnBitUtils::WriteTo(x, numThmTrigs);
        x = SnBitUtils::WriteTo(x, numSavedEvts);
        x = SnBitUtils::WriteTo(x, seqlive);
        
        return SnCommWin::kOkMsgSent;
    }

    static
    uint32_t SizeOf(const uint8_t rv, const uint32_t confLblLen,
                    const uint8_t loseLSB, const uint8_t loseMSB) {
        // number of bytes read/written during i/o
        const uint32_t maxsize = GetMaxSizeOf(rv);
        const uint32_t msz = maxsize - SnConfigFrame::kConfLblLen 
                           + confLblLen;
        if (rv==1) {
            if ((loseLSB==0) && (loseMSB==0)) {
                return msz;
            } else {
                return msz - maxsize
                           + SnEventFrame::SizeOf(SnEventFrame::kIOVers,
                                                  loseLSB, loseMSB);
            }
        } else {
            return msz;
        }
    }
    
    static
    uint32_t SizeOf(const uint8_t rv, const SnConfigFrame& conf) {
        return SizeOf(rv, conf.GetLabelStrLen(),
                      conf.GetWvLoseLSB(), conf.GetWvLoseMSB());
    }
    
};

#endif // SN_SnStatusFrame
