#ifndef SN_SnCommWin
#define SN_SnCommWin

#include "mbed.h"
#include <stdint.h>
#include <string>

#include "SnConfigFrame.h"
class SnEventFrame;
class SnPowerFrame;
class SnSignalStrengthFrame;
class SnCommPeripheral;
class SnTempFrame;
class SnClockSetFrame;
class SnHeartbeatFrame;

// DAQ station communication utilities

class SnCommWin {
 public:
    enum ECommWinResult {
        kUndefFail,     // undefined fail type
        kCanNotConnect, // unable to connect to port
        kFailTimeout,   // timed out but message required
        kFailNoneSent,  // none of the message sent
        kFailPartSent,  // only part of the message sent
        kUnexpectedRec, // unexpected / unhandled message received
        kAllFails,      // to be used for "if (ret > kAllFails) ==> success"
        // only successes should go below
        kConnected,     // connection established, no messaging attempted
        kOkStopComm,    // received signal to stop communicating
        kOkMsgSent,     // message sent without waiting for "received" handshake
        kOkNoMsg,       // timed out with no message, but that is ok
        kOkWithMsg,     // successfully received message
        kOkWthMsgNoConf,// successfully received message and it says to stick with the same config
        kOkWthMsgDidDel // successfully received message and deleted a file because of it
    };

    static const char*    kLocalDir;          // the local mbed directory
    static const char*    kDelAllConfCodeStr; // a magic string to confirm deletion of all files on the SD card
    static const uint8_t  kDelAllConfCodeLen; // the length of the magic string
 
 private:
    bool                  fSendingInHandshake; // true if we are sending files as a result of a handshake.
                                               // will ignore further handshake requests to send new files
                                               // in order to avoid a stack overflow
    
    
    SnCommWin::ECommWinResult GetFilename(const uint32_t timeout,
                                          char* const buf,
                                          const uint32_t namelen);
    bool                      BuildLocalFileName(std::string fname,
                                                 const char* const dir,
                                                 std::string& lfname);
    SnCommWin::ECommWinResult GetLocalFile(std::string fname,
                                           char* const buf,
                                           const uint32_t bsize,
                                           const uint32_t timeout,
                                           std::string& lfname);
    int16_t                   LoopLocalDirBinFiles(const bool doRemove,
                                                   const std::string& fname);
    SnCommWin::ECommWinResult GetHeader(const uint32_t timeOut,
                                        char* const buf,
                                        const uint32_t bsize,
                                        uint8_t& mcode,
                                        uint32_t& mlen);

    bool WriteStatusDataPackHeaderTo(char*& b, 
                                     const uint32_t payloadLen);

    bool CallHeaderWriteTo(char*& b,
                           const uint8_t msgCode,
                           const uint32_t msgLen);
    
    int32_t CallSendAll(const char* const data, const uint32_t length,
                        const uint32_t timeout_clock);


    template<class T>
    char* WriteStatusDataPackHeadersFor(const T& x,
                                        const uint8_t hdrCode,
                                        char*& b,
                                        const uint32_t timeout_clock) {
        // flag that this is a data packet
        const uint32_t xMsgLen = x.SizeOf();
        WriteStatusDataPackHeaderTo(b, xMsgLen);
        CallHeaderWriteTo(b, hdrCode, xMsgLen);
        return b;
    }

    template<class T>
    char* SendStatusDataPack(const T& x,
                             const uint8_t hdrCode,
                             int32_t& toBeSent,
                             int32_t& byteSent,
                             char* const genBuf,
                             const uint32_t timeout_clock) {
        char* b = genBuf;
        b = WriteStatusDataPackHeadersFor(x, hdrCode, 
                                          b, timeout_clock);
        x.WriteTo(b);
        const int32_t sendNow = (b - genBuf);
        toBeSent += sendNow;
        byteSent += CallSendAll(genBuf, sendNow, timeout_clock);
        return b;
    }
    

 protected:
    SnCommPeripheral*           fComm; // the communication peripheral. deleted in dtor!!

    virtual int32_t SendFileBlock(FILE* inf, 
                                  const uint8_t blockHeaderCode,
                                  const uint32_t blockSize,
                                  char* const genBuf,
                                  int32_t& bytesToSend,
                                  const uint32_t timeout);

    virtual int32_t SendFileContents(FILE* inf,
                                     const SnConfigFrame& curConf,
                                     SnEventFrame& evt,
                                     SnPowerFrame& pow,
                                     char* const genBuf,
                                     uint32_t nevts,
                                     int32_t& bytesToBeSent,
                                     const uint32_t timeout_clock);
    
 public:
    SnCommWin(SnCommPeripheral* p);
    virtual ~SnCommWin();
    
    // probably no need to overload
    virtual bool            TrySetSysTimeUnix(const uint32_t timeout,
                                              uint32_t& prvTime,
                                              uint32_t& setTime);
    virtual bool            Connect(const uint32_t timeout);
    virtual bool            CloseConn(const uint32_t timeout,
                                      char* const genBuf=0,
                                      const bool sendCloseSignal=false);
    virtual bool            PowerDown(const uint32_t timeout);
    
    // optional overloads
    virtual void Set(const SnConfigFrame& conf) {}
    
    // mandatory overloads
    virtual SnConfigFrame::EDatPackBit GetCommType() const=0;
    
    virtual ECommWinResult OpenWindow(const bool sendStatus,
                                      const SnConfigFrame& conf,
                                      const SnPowerFrame& pow, // com win power
                                      const SnEventFrame& stEvent,
                                      const uint16_t seq,
//                                      const float thmrate,
//                                      const float evtrate,
                                      const uint32_t numThmTrigs,
                                      const uint32_t numSavedEvts, 
                                      const float    seqlive,
                                      const uint32_t powerOnTime,
                                      const SnTempFrame& temper, // com win temp
                                      char* const genBuf,
                                      const uint32_t timeout_clock)=0;

    virtual bool            GetDeleteAllConfirmCode(const SnConfigFrame& conf,
                                                    const uint32_t length,
                                                    const uint32_t timeout,
                                                    char* const buf,
                                                    const uint32_t bsize);
    virtual ECommWinResult  WaitHandshake(const SnConfigFrame& conf,
                                          const uint32_t timeout,
                                          char* const buf,
                                          const uint32_t bsize,
                                          uint8_t& hndShkCode,
                                          uint32_t* hndShkLen=0);

    virtual ECommWinResult  HandleHandshake(FILE* inf, const char* infn,
                                            const SnConfigFrame& curConf,
                                            SnEventFrame& evt,
                                            SnPowerFrame& pow,
                                            char* const genBuf,
                                            const uint32_t bsize,
                                            const uint32_t handshakeTimeout,
                                            const uint8_t hndshk,
                                            const uint32_t hndlen,
                                            const uint32_t* nevts=0);
    int32_t                 SendHndshkReq(char* const genBuf,
                                          const uint32_t timeout_clock);
    
    ECommWinResult  SendSignalStrength(char* const genBuf,
                                       SnSignalStrengthFrame& sigstr,
                                       const uint32_t timeout_clock);

    ECommWinResult  GetConfig(SnConfigFrame& conf,
                              const uint32_t timeOut,
                              char* const confBuf,
                              const uint32_t bsize);
    
    ECommWinResult  SendStatus(const SnConfigFrame& conf,
                               const SnPowerFrame& pow, // com win power
                               const SnEventFrame& stEvent,
                               const uint16_t seq,
//                               const float thmrate,
//                               const float evtrate,
                               const uint32_t numThmTrigs,
                               const uint32_t numSavedEvts, 
                               const float    seqlive,
                               const uint32_t powerOnTime,
                               const SnTempFrame& temper, // com win temp
                               char* const genBuf,
                               const uint32_t timeout_clock);

    virtual ECommWinResult SendStatusData(const SnConfigFrame& conf,
                                          const SnConfigFrame& stConf,
                                          const SnClockSetFrame& stTrgStartClk,
                                          const SnClockSetFrame& stTrgStopClk,
                                          const SnPowerFrame& stPower,
                                          const SnEventFrame& stEvent,
                                          const SnTempFrame& stTemperature,
                                          const SnHeartbeatFrame& stHeartbeat,
                                          const bool stNewPower,
                                          const bool stNewEvent,
                                          const bool stNewHeartbeat,
                                          const bool stNewTemperature,
                                          char* const genBuf,
                                          const uint32_t timeout_clock);
    
    SnCommWin::ECommWinResult SendString(const char* str,
                                         const uint32_t timeout);

    int32_t SendFilename(const char* fn,
                         char* const genBuf,
                         int32_t&    bytesToBeSent,
                         const uint32_t timeout_clock);

    ECommWinResult SendDataFromFile(FILE* inf, const char* infn,
                            const SnConfigFrame& curConf,
                            SnEventFrame& evt,
                            SnPowerFrame& pow,
                            char* const genBuf,
                            const uint32_t bsize,
                            const uint32_t nevts,
                            const uint32_t timeout_clock,
                            uint8_t* hndcode=0,
                            uint32_t* hndlen=0);

    ECommWinResult  SendData(SnConfigFrame& conf,
                             SnEventFrame& evt,
                             SnPowerFrame& pow,
                             char* const genBuf,
                             const uint32_t bsize,
                             const uint32_t timeout);
    
    // assume little endian
    static
    uint16_t GetMinSeqFrom(const uint32_t hndshk) {
        // left-most 2 bytes
        return (hndshk>>16u);
    }
    static
    uint16_t GetMaxSeqFrom(const uint32_t hndshk) {
        // right-most 2 bytes
        return (hndshk&0xFFFF);
    }
    
};

#endif // SN_SnCommWin
