#ifndef SN_SnBitUtils
#define SN_SnBitUtils

#include "SnProtocolConstants.h"

#include "mbed.h"
#include <stdint.h>

//
// utilities for reading/writing bytes
//
#define BIT(n)               (1ULL << (n))
#define BITS_IN_SHORT        (16)
#define BITS_IN_CHAR         (8u)

struct SnBitUtils {
    /* save memory: hardcode the values instead
    static const uint8_t kBitsInShort = sizeof(int16_t)*8;
    static const uint8_t kShrtsInInt  = sizeof(int32_t)/sizeof(int16_t);
    static const uint8_t kCharsInInt  = sizeof(int32_t)/sizeof(int8_t);
    */

#ifdef PROT_SUPPORT_DCARDS
    static
    void SetChanNumBits(const uint8_t ch,
                        DigitalOut& hibit, DigitalOut& lobit) {
        // ch = 0,1,2,3
        // extract its bits
        hibit = (ch & 0x0002u) >> 1u;
        lobit = (ch & 0x0001u);
    }
    
    static
    uint8_t binToGray(const uint8_t x) {
        return (x >> 1u) ^ x;
    }
#endif // PROT_SUPPORT_DCARDS    

#ifdef PROT_SUPPORT_SERIAL
    //
    // USB byte I/O
    //
    template<typename T>
    static
    Serial& ReadFrom(Serial& ser, T& x) {
        // little endian!
        x=0;
        for (uint8_t i=0; i<sizeof(T); i++) {
            x |= ser.getc() << (i*8u);
        }
        return ser;
    }
    
    template<typename T>
    static
    Serial& WriteTo(Serial& ser, const T& x) {
        // little endian!
        for (uint8_t i=0u; i<sizeof(T); i++) {
            ser.putc( x >> (i*8u) );
        }
        return ser;
    }

    static
    Serial& ReadFrom(Serial& ser, 
                        char* const x, const uint32_t nbytes) {
        char* buf = x;
        for (uint32_t i=0; i<nbytes; i++, buf++) {
            *buf = ser.getc();
        }
        return ser;
    }
    
    static
    Serial& WriteTo(Serial& ser,
                       const char* const x, const uint32_t nbytes) {
        const char* buf = x;
        for (uint32_t i=0u; i<nbytes; i++, buf++) {
            ser.putc(*buf);
        }
        return ser;
    }
#endif // PROT_SUPPORT_SERIAL
    
    //
    // memory buffer byte I/O
    //
    template<typename T>
    static
    const char* ReadFrom(const char* const buf, T& x) {
        // not a string to int conversion, but a byte copy
        memcpy(&x, buf, sizeof(T));
        return buf+sizeof(T);
    }
    
    template<typename T>
    static
    char* WriteTo(char* const buf, const T x) {
        // not an int to string conversion, but a byte copy
        memcpy(buf, &x, sizeof(T));
        return buf+sizeof(T);
    }
    
    template<typename T>
    static
    const char* ReadFrom(const char* const buf, T* const x,
                         const uint32_t slen) {
        // just a byte copy
        memcpy(x, buf, slen*sizeof(T));
        return buf+slen;
    }
    
    template<typename T>
    static
    char* WriteTo(char* const buf, const T* const x,
                  const uint32_t slen) {
        // just a byte copy
        memcpy(buf, x, slen*sizeof(T));
        return buf+slen;
    }

#ifdef PROT_SUPPORT_FILE    
    //
    // file byte I/O
    //
    template<typename T>
    static
    FILE* ReadFrom(FILE* f, T& x) {
        fread(&x, sizeof(T), 1u, f);
        return f;
    }
    
    template<typename T>
    static
    FILE* WriteTo(FILE* f, const T x) {
        fwrite(&x, sizeof(T), 1u, f);
        return f;
    }
    
    static
    FILE* ReadFrom(FILE* f, char* const x,
                   const uint32_t slen) {
        fread(x, sizeof(char), slen, f);
        return f;
    }
    
    static
    FILE* WriteTo(FILE* f, const char* const x,
                  const uint32_t slen) {
        fwrite(x, sizeof(char), slen, f);
        return f;
    }
#endif // PROT_SUPPORT_FILE
   
    template<typename T>
    static
    void printBits(const T x, const bool endLine) {
        for (int32_t i=(sizeof(T)*BITS_IN_CHAR)-1; i>-1; i--) {
            if ( (x&(1<<i))==0 ) {
                printf("0");
            } else {
                printf("1");
            }
        }
        if (endLine) {
            printf("\r\n");
        }
    }

};
#endif // SN_SnBitUtils