#include "SnCommSBD.h"

#ifdef ENABLE_SBD


#define DEBUG

#include <stdint.h>
#include <vector>

#ifdef USE_MODSERIAL
#include "MODSERIAL.h"
#endif

#include "SnBitUtils.h"
#include "SnCommConstants.h"

const uint16_t SnCommSBD::kMaxSend;
const uint16_t SnCommSBD::kMaxRecv;
const uint16_t SnCommSBD::kWaitOnGSS;
const uint16_t SnCommSBD::kRxBufSz;

int32_t SnCommSBD::ReceiveAll(char* const buf, const uint32_t mlen,
                              const uint32_t timeout_clock) {
#ifdef DEBUG
    printf("SnCommSBD::ReceiveAll : mlen=%u\r\n",mlen);
#endif
    //return DoAllIO(buf, mlen, timeout_clock, kMaxRecv, &SnCommSBD::RecvSBD);
    return DoIO(buf, mlen, timeout_clock, &SnCommSBD::RecvSBD);
}

int32_t SnCommSBD::SendAll(const char* const data, const uint32_t length,
                           const uint32_t timeout_clock) {
#ifdef DEBUG
    printf("SnCommSBD::SendAll : length=%u\r\n",length);
#endif
    fSMsgTot = length / kMaxSend;
    if ( (length%kMaxSend)>0 ) {
        ++fSMsgTot;
    }
    fSMsgNum = 0;
    /*
    return DoIO<const char* const>(data, length, timeout_clock,
                                   &SnCommSBD::SendBufferedSBD);
    */
    return SendBufferedSBD(data, length, timeout_clock);
}

bool SnCommSBD::SerialReadable() {
    return (fSBD!=0) && (fSBD->readable()!=0);
}

bool SnCommSBD::SerialWriteable() {
    return (fSBD!=0) && (fSBD->writeable()!=0);
}

int32_t SnCommSBD::PutC(const char* const data, const uint32_t,
                        const uint32_t) {
    // send a single character to the SBD serial port
    // extra parameters allow it to be called by DoIO
    if (fSBD!=0) {
        fSBD->putc(*data);
        return 1;
    }
    return 0;
}

int32_t SnCommSBD::GetC(char* const data, const uint32_t,
                        const uint32_t) {
    // receive a single character from the SBD serial port
    // extra parameters allow it to be called by DoIO
    if (fSBD!=0) {
        // only do one byte at a time, so DoIO will keep
        // checking that we're readable
        *data = fSBD->getc();
        return 1;
    }
    return 0;
}

int32_t SnCommSBD::FinishSending(const uint32_t timeout_clock) {
    int32_t b=0;
    while (fTxPos>b) {
        if (IsTimedOut(timeout_clock)) {
            break;
        }
        b += SendSBD(fTxBuf+b, fTxPos-b, timeout_clock);
    } // else let it go around and try again

    // without another variable (like RxPos and RxSiz) to
    // keep track of where to start and stop reading from the buffer,
    // if a partial message was sent, we will lose the rest.
    // (since we always read from byte 0 of the buffer)
    // if this ever happens, we should change this routine.
    //
    // in the meantime, calls to FinishSending will
    // always clear the outgoing buffer
    fTxPos = 0;
    
    return b;
}

int32_t SnCommSBD::SendBufferedSBD(const char* const data,
                                   const uint32_t length,
                                   const uint32_t timeout_clock) {
    // here we assume that fTxPos and kTxBufSz are smaller than the max
    // message size SBD can accept (kMaxSend). but this is NOT checked
    //
    // returns the number of bytes sent out by SendSBD, not the number
    // of bytes copied to the buffer, which means this function
    // CANNOT be sent to DoIO
    uint32_t b=0, bsent=0;
    const char* d = data;
    char*       s = &(fTxBuf[fTxPos]);
    while (length>b) {
        if (IsTimedOut(timeout_clock)) {
            break;
        }
        if (fTxPos<kTxBufSz) {
            // copy this byte to the outgoing buffer
            // and increment counters
            *s = *d;
            ++s; ++d; ++fTxPos; ++b;
        } else {
            // the outgoing buffer is full
            const int32_t bs = SendSBD(fTxBuf, fTxPos, timeout_clock);
            if (bs==fTxPos) {
                // sent successfully. reset outgoing buffer
                s = fTxBuf;
                fTxPos = 0;
                bsent += bs;
            } // else let it go around and try again
        }
    }
    return bsent;
}

int32_t SnCommSBD::SendSBD(const char* const data, const uint32_t length, 
                           const uint32_t timeout_clock) {
    // send a short burst binary message
    // length should be <=kMaxSend, but this is not checked
    // (and kMaxSend should allow us room for the SbdMsg header!)

#ifdef DEBUG
    printf("SnCommSBD::SendSBD : length = %u\r\n",length);
#endif

    const uint16_t origMsgNum = fSMsgNum;

    // initiate SBD write binary
    char sbdwb[32];
    sprintf(sbdwb, "AT+SBDWB=%lu", length+SnHeaderFrame::SizeOf());
    AtCmd(sbdwb, timeout_clock);
    
#ifdef DEBUG
    printf("waiting for READY\r\n");
#endif
    // wait for ready signal
    if (WaitForAtResponse("READY",timeout_clock)) {
        // send the data
        
        
        // first append the SbdMsg header
        ++fSMsgNum; // first msg will be #1, not #0
        char* smhead = SnHeaderFrame::GetHdBuf(
            SnHeaderFrame::kSbdMsgCode,
            GetHeaderFromMsgNumTot(true));
        
        // calculate the checksum
        union {
            uint32_t u;
            char     c[sizeof(uint32_t)];
        } cs;
        cs.u = 0;
        const char* d = smhead;
        for (uint32_t i=0; i<SnHeaderFrame::SizeOf(); ++i, ++d) {
            cs.u += *d;
        }
        d = data;
        for (uint32_t i=0; i<length; ++i, ++d) {
            cs.u += *d;
        }
        uint32_t bsent(0);
        bsent = DoIO(smhead, SnHeaderFrame::SizeOf(), timeout_clock,
             &SnCommSBD::PutC, &SnCommSBD::SerialWriteable);
#ifdef DEBUG
        printf("SendSBD: header DoIO returned %u\r\n", bsent);
#endif
        bsent = DoIO(data, length, timeout_clock,
             &SnCommSBD::PutC, &SnCommSBD::SerialWriteable);
#ifdef DEBUG
        printf("SendSBD: data DoIO returned %u\r\n", bsent);
#endif
        // now least significant bytes of the checksum (in big-endian order)
        bsent = DoIO((cs.c)+1, 1u, timeout_clock,
             &SnCommSBD::PutC, &SnCommSBD::SerialWriteable);
#ifdef DEBUG
        printf("SendSBD: checksum byte 1 DoIO returned %u\r\n", bsent);
#endif
        bsent = DoIO( cs.c,    1u, timeout_clock,
             &SnCommSBD::PutC, &SnCommSBD::SerialWriteable);
#ifdef DEBUG
        printf("SendSBD: checksum byte 0 DoIO returned %u\r\n", bsent);
#endif
        //wait_ms(kCmdReactDelay);
        
#ifdef DEBUG
        printf("sbd message:\r\n");
        const char* dd = smhead;
        for (uint32_t i=0; i<SnHeaderFrame::SizeOf(); ++i, ++dd) {
            printf("%02x ",*dd);
        }
        dd = data;
        for (uint32_t i=0; i<length; ++i, ++dd) {
            printf("%02x ",*dd);
        }
        printf("%02x %02x\r\n",cs.c[1],cs.c[0]);
#endif
        if (WaitForAtResponse("0", timeout_clock)) {
            // modem got the bytes. send the message.
            AtCmd("AT+SBDIX", timeout_clock);

            std::string ix;
            if (WaitForAtResponse("+SBDIX:",timeout_clock,ix)) {

                int32_t mo, momsn, mt, mtmsn, mtlen, mtq;
                if ( ParseSBDIXresp(ix,
                        mo, momsn, mt, mtmsn, mtlen, mtq) ) {
                    
                    // clear the SBD's outgoing buffer
                    AtCmd("AT+SBDD0", timeout_clock);
                    WaitForAtResponses("0;1", timeout_clock);
                
                    // check if the message was sent successfully
                    if (mo<5) {
                        // message sent successfully
#ifdef DEBUG
                        printf("return length %d\r\n",
                            static_cast<int32_t>(length));
#endif
                        return static_cast<int32_t>(length);
                    }
                }
            }
        }
        
        // '1' => insufficient bytes sent to SBD
        // '2' => checksum mismatch
        // '3' => length <1 byte or >340 bytes
        
    }

#ifdef DEBUG
    printf("reutrn 0\r\n");
#endif
    // something went wrong. put the msg number back
    // and return 0 bytes sent.
    fSMsgNum = origMsgNum;
    return 0;
}

int32_t SnCommSBD::RecvSBD(char* const data, const uint32_t length, 
                           const uint32_t timeout_clock) {
    // get data from a short burst binary message
    // length should be <=kMaxSend, but this is not checked
    //
    // we allow for sevearl possibilities:
    // (a) this call only wants a fraction of the data contained in
    //     the full SBD message (i.e. it's only reading a header)
    //     .. for this, we use a local buffer
    // (b) the data may be spread over multiple SBD messages
    //     .. in this case, the messages must arrive in order or
    //     the out of order data will not be copied to 'data'
    //     .. for this, we use the SBD header and fRMsgNum, fRMsgTot
    
    
#ifdef DEBUG
    printf("SnCommSBD::RecvSBD : length = %u. "
        "fRxPos=%hu, fRxSiz=%hu, fRMsgNum=%hu, fRMsgTot=%hu\r\n",
        length, fRxPos, fRxSiz, fRMsgNum, fRMsgTot);
#endif
    
    char* dat  = data;
    int32_t b = 0;
    if ( (fRxPos>0) && (fRxSiz>fRxPos) ) {
        // want the data from the last read (first)
        const uint32_t maxlen = (fRxSiz-fRxPos)*sizeof(char);
        const uint32_t mlen   = (length<maxlen) ? length : maxlen;
#ifdef DEBUG
        printf("copying %u bytes from fRxBuf\r\n", mlen);
#endif
        memcpy(dat, fRxBuf+fRxPos, mlen);
        b   += mlen;
        dat += b;
        // update buffer position
        fRxPos += mlen;
        if (fRxPos>=fRxSiz) {
            // done with local buffer
            fRxPos = fRxSiz = 0;
        }
    }

#ifdef DEBUG
    printf("fRxPos=%hu, fRxSiz=%hu\r\n",fRxPos,fRxSiz);
    printf("b=%d, length=%u\r\n",b,length);
#endif
    
    if (b<length) { // note 'if' not 'while'
        // if we get in here, the local rx buffer has been read completely
        // so fRxPos==0 -- this is exploited by always writing the SBD message
        // to the start of fRxBuf, without checking the value of fRxPos.
        
        // initiate session and check for messages
        // this will bring a message into the modem if there is one
        AtCmd("AT+SBDIX", timeout_clock);

        std::string ix;
        if (WaitForAtResponse("+SBDIX:",timeout_clock,ix,"OK\r\n")) {
            
            int32_t mo, momsn, mt, mtmsn, mtlen, mtq;
            if ( ParseSBDIXresp(ix,
                    mo, momsn, mt, mtmsn, mtlen, mtq) ) {
                
                if (mt!=1) {
                    // wait a bit. then just return. DoIO should cause
                    // this function to be called again for another try
#ifdef USE_RTOS
                    Thread::wait(kWaitOnGSS*1000);
#else
                    wait(kWaitOnGSS);
#endif
                } else {
                    
                    // get (the rest of) the data from an SBD message.
                    // do some setup stuff before asking for the message
                    // in case we're not using buffered serial (altho it
                    // probably won't work in that case anyway)
                    
                    // ensure the RX buffer is totally empty
                    // before asking for the binary message
                    EmptyRxBuffer(timeout_clock);
                    
                    // pull the message from the modem
                    AtCmd("AT+SBDRB", timeout_clock);

#ifdef DEBUG
                    printf("getting sbd msg len\r\n");
#endif
                    // get the 2-byte message length (in big-endian order)
                    char c[2];
                    DoIO(c, 2u, timeout_clock,
                         &SnCommSBD::GetC, &SnCommSBD::SerialReadable);
                    uint16_t ts = (c[0]<<BITS_IN_CHAR)+c[1];
#ifdef DEBUG
                    printf("msg len .x%02x..x%02x. = %hu\r\n",
                        c[0], c[1], ts);
#endif
                    

                    if (ts >= SnHeaderFrame::SizeOf()) {
                        // to store the SbdMsg header
                        char hbuf[SnHeaderFrame::SizeOf()];
                        // get the header
                        DoIO(hbuf, SnHeaderFrame::SizeOf(), timeout_clock,
                             &SnCommSBD::GetC, &SnCommSBD::SerialReadable);
#ifdef DEBUG
                        printf("Got header: ");
                        dispStrBytes(hbuf, SnHeaderFrame::SizeOf());
                        printf("\r\n");
#endif
                        // copy the message to the rx buffer
                        // without the SbdMsg header
                        ts -= SnHeaderFrame::kMaxSizeOf;
#ifdef DEBUG
                        printf("getting remaining %hu bytes\r\n",ts);
#endif
                        // always copy to the start of the buffer, so this
                        // check is sufficient to prevent buffer overflow.
                        // note that if we're here, fRxPos is 0 anyway
                        if (ts<=kMaxRecv) {
                           fRxSiz = DoIO(fRxBuf, ts, timeout_clock,
                                          &SnCommSBD::GetC, &SnCommSBD::SerialReadable);

                            // get the 2-byte checksum
                            char rxcsb[2];
                            const int rxcsSize = DoIO(rxcsb, 2, timeout_clock,
                                                      &SnCommSBD::GetC,
                                                      &SnCommSBD::SerialReadable);
                            const uint32_t rxcs = (rxcsb[0]<<BITS_IN_CHAR)+rxcsb[1];
                            // calculate the checksum
                            uint32_t cs(0);
                            const char* d = hbuf;
                            for (uint16_t i=0; i<SnHeaderFrame::SizeOf(); ++i, ++d) {
                                cs += *d;
                            }
                            d = fRxBuf;
                            for (uint16_t i=0; i<ts; ++i, ++d) {
                                cs += *d;
                            }
#ifdef DEBUG
                            printf("put into fRxBuf:\r\n");
                            dispStrBytes(fRxBuf, ts);
                            printf("\r\n");
                            printf("rxcsb[0]=%#02x, rxcsb[1]=%#02x, rxcs=%u, cs=%u\r\n",
                                rxcsb[0], rxcsb[1], rxcs, cs);
#endif
                            if (rxcs == cs) {
                                
                                // serial i/o done. now do the work
                                
                                // it better be an SbdMsg!
                                uint8_t hc; uint32_t hl;
                                const char* rxb = hbuf;
                                SnHeaderFrame::ReadFrom(rxb, hc, hl);
#ifdef DEBUG
                                printf("hc = .x%02x., hl = %u (%hu/%hu)\r\n",
                                    hc, hl, GetMsgTotFromHeader(hl),
                                    GetMsgNumFromHeader(hl));
#endif
                                if (hc==SnHeaderFrame::kSbdMsgCode) {
                                    // figure out if this is the msg we want
                                    bool saveMsg = false;
                                    const uint16_t tot = GetMsgTotFromHeader(hl);
                                    if (fRMsgTot==0) {
                                        // haven't saved a message yet.
                                        if ( mtq<=(tot-1) ) {
                                            // if the total messages (minus this one)
                                            // is equal to or more than what's in the queue,
                                            // then this message is the start of our bunch
                                            saveMsg=true;
                                        }
                                        // else skip this message and get a newer one
                                    } else if (tot==fRMsgTot) {
                                        // we've already gotten the first X messages
                                        // of the bunch
                                        const uint16_t mnum = GetMsgNumFromHeader(hl);
                                        if (mnum==(fRMsgNum+1)) {
                                            saveMsg=true;
                                        }
                                        // else this bunch is screwed and we will eventually
                                        // time out, without using this out of order data
                                    }
                                    // else this bunch is out of order. skip it
#ifdef DEBUG
                                    printf("fRMsgTot=%hu, fRMsgNum=%hu, tot=%hu, "
                                        "mtq=%d, mnum=%hu, save=%s\r\n",
                                        fRMsgTot, fRMsgNum, tot, mtq,
                                        GetMsgNumFromHeader(hl),
                                        saveMsg?"true":"false");
#endif                            
                                    if (saveMsg) {
                                        GetMsgNumTotFromHeader(hl, fRMsgNum, fRMsgTot);
                                        // copy the data to the external buffer
#ifdef DEBUG
                                        printf("length=%u, b=%d, ts=%hu\r\n",
                                            length, b, ts);
#endif
                                        if ((length-b)<ts) {
                                            ts    = length-b; // copy up to here
                                            fRxPos = ts; // still some data remaining
                                            // fRxSiz is at the end of this SBD message
                                            // set by DoIO return value
                                        } else {
                                            fRxPos = 0;   // read whole buffer
                                            fRxSiz = 0;
                                        }
                                        memcpy(dat, fRxBuf, (ts)*sizeof(char));
#ifdef DEBUG
                                        printf("fRxPos=%hu, fRxSiz=%u\r\n",
                                            fRxPos, fRxSiz);
#endif
                                        
                                        // not nec. the full SBD message (anymore)
                                        b = static_cast<int32_t>(ts);
                                        
                                        // are we done?
                                        if (fRMsgNum==fRMsgTot) {
                                            fRMsgNum=fRMsgTot=0;
                                            // there might still be data to be
                                            // read in the local buffer, but
                                            // there are no more SBD messages
                                        }
#ifdef DEBUG
                                        printf("fRMsgNum=%hu, fRMsgTot=%hu\r\n",
                                            fRMsgNum, fRMsgTot);
#endif
                                    } // saveMsg?
                                } // header code is SBDMsgCode?
                            } // checksum ok?
                        } else {
                            // message is too long for our buffer!
                            // send a warning
                            char tmp[kMaxStrLen];
                            sprintf(tmp, "Message %d too long (%hu) to read. "
                                "Max=%hu.",
                                mtmsn, ts, kMaxRecv);
                            SendString(tmp, timeout_clock);
                        }
                    } // msg length able to hold a SnHeaderFrame?
                    // else: just return without using the message
                    // DoIO should call this function again, and we will obtain
                    // the next message. the cycle should continue until the
                    // buffer is clear and all old message are thrown away
                    
                    // clear the SBD incoming buffer
                    AtCmd("AT+SBDD1", timeout_clock);
                    WaitForAtResponses("0;1", timeout_clock);
                }
            }
        }
    }
#ifdef DEBUG
    printf("return b=%d\r\n",b);
#endif
    return b;
}

void SnCommSBD::EmptyRxBuffer(const uint32_t timeout) {
#ifdef DEBUG
    printf("EmptyRxBuffer\r\n");
#endif
    while ( SerialReadable() ) {
        if ( IsTimedOut(timeout) ) {
            break;
        }
        fSBD->getc(); // throw away
    }
}

bool SnCommSBD::ParseSBDSresp(const std::string& ixr,
                              int32_t& mo, int32_t& momsn,
                              int32_t& mt, int32_t& mtmsn) {
    const size_t start = ixr.find("SBDS:");
    if (start!=std::string::npos) {
        const int n = sscanf(ixr.data()+start,
           "SBDS:%d,%d,%d,%d",
           &mo, &momsn, &mt, &mtmsn);
/*
#ifdef DEBUG
        printf("n=%d, mo=%d, momsn=%d, mt=%d, mtmsn=%d, "
            "mtlen=%d, mtq=%d\r\n",
            n, mo, momsn, mt, mtmsn, mtlen, mtq);
#endif
*/
        return (n==4);
    }
    return false;
}

bool SnCommSBD::ParseSBDIXresp(const std::string& ixr,
                               int32_t& mo, int32_t& momsn,
                               int32_t& mt, int32_t& mtmsn,
                               int32_t& mtlen, int32_t& mtq) {
/*
#ifdef DEBUG
    printf("ixr: %s\r\n",ixr.c_str());
#endif
*/
    const size_t start = ixr.find("SBDIX:");
    if (start!=std::string::npos) {
        const int n = sscanf(ixr.data()+start,
           "SBDIX:%d,%d,%d,%d,%d,%d",
           &mo, &momsn, &mt, &mtmsn, &mtlen, &mtq);
/*
#ifdef DEBUG
        printf("n=%d, mo=%d, momsn=%d, mt=%d, mtmsn=%d, "
            "mtlen=%d, mtq=%d\r\n",
            n, mo, momsn, mt, mtmsn, mtlen, mtq);
#endif
*/
        return (n==6);
    }
    return false;
}

int32_t SnCommSBD::AtCmd(const char* const cmd,
                         const uint32_t timeout) {
    // first flush the serial read buffer 
    // to remove any responses to previous commands

    int32_t br=0;
    if (fSBD!=0) {
#ifdef DEBUG
        printf("AtCmd ");
#endif
        const uint32_t cln = strlen(cmd);
#ifdef DEBUG
        dispStrBytes(cmd, cln);
#endif
        br  = DoIO(cmd, cln, timeout,
                   &SnCommSBD::PutC,
                   &SnCommSBD::SerialWriteable);
        if ( cmd[cln-2]!='\r' || cmd[cln-1]!='\n' ) {
            // static char rr[2]={0x0d, 0x0a};
            static char rr[1]={0x0d}; // do not send a '\n', suggested by NAL
#ifdef DEBUG
            // dispStrBytes(rr, 2u);
            dispStrBytes(rr, 1u);
#endif
            // br += DoIO(rr, 2u, timeout,
            //           &SnCommSBD::PutC,
            //           &SnCommSBD::SerialWriteable);
            br += DoIO(rr, 1u, timeout,
                       &SnCommSBD::PutC,
                       &SnCommSBD::SerialWriteable);
        }
#ifdef DEBUG
        printf("\r\n");
#endif
    }
    return br;
}

int32_t SnCommSBD::AtResp(char* const res, const uint32_t rlen,
                          const uint32_t timeout) {
#ifdef DEBUG
//    printf("AtResp\r\n");
#endif
    return DoIO(res, rlen, timeout,
                &SnCommSBD::GetC, &SnCommSBD::SerialReadable);
}

void SnCommSBD::AppendRespTxt(std::string& str,
                              const uint32_t timeout,
                              const uint32_t maxslen) {
#ifdef DEBUG
//    printf("AppendRespTxt\r\n");
#endif
    char c(0);
    
    while (SerialReadable()) {
#ifdef DEBUG
//    printf("AppendRespTxt readable\r\n");
#endif
        if (IsTimedOut(timeout)) {
            break;
        }
        
        GetC(&c, 1, timeout);
        if (str.length()>=maxslen) {
            // drop the first char
            str.erase(str.begin());
        }
        str += c; // keep it
    }
}

bool SnCommSBD::AtRespContainsTxt(const char* const desired,
                                  const uint32_t timeout,
                                  std::string& str,
                                  const char* until,
                                  const uint32_t maxslen) {
    // searches for the desired text in the AT response
    // if found, saves text from that point until the terminator
    // 'until' (usually "\r\n")
    AppendRespTxt(str, timeout, maxslen);
#ifdef DEBUG
//    printf("AtRespContainsTxt: str=%s, desired=%s\r\n", str.c_str(), desired);
#endif
    const size_t idx = str.rfind(desired);
    const bool found = (idx!=std::string::npos);
    if (found) {
        size_t iend = str.find(until,idx);
        // get the rest of the response if necessary
        while ( (iend==std::string::npos)
             && (IsTimedOut(timeout)==false) ) {
            AppendRespTxt(str, timeout, maxslen);
            iend = str.find(until,idx);
        }
        // drop the characters before the desired response
        // keep characters after the first 'until', in case
        // the start of another message is in here
        str = str.substr(idx,std::string::npos);
    }
    return found;
}

bool SnCommSBD::WaitForAtResponse(const char* const desired,
                                  const uint32_t timeout,
                                  const char* until) {
    std::string str;
    return WaitForAtResponse(desired, timeout, str, until);
}

bool SnCommSBD::WaitForAtResponse(const char* const desired,
                                  const uint32_t timeout,
                                  std::string& str,
                                  const char* until) {
#ifdef DEBUG
    printf("WaitForAtResponse (%s)\r\n",desired);
#endif
    bool gotit = false;
    do {
        gotit = AtRespContainsTxt(desired,timeout,str,until);
    } while ((gotit==false)
           && IsTimedOut(timeout)==false);
#ifdef DEBUG
    printf("WaitForAtResponse: returning %s (",
        gotit?"true":"false");
    dispStrBytes(str.c_str(),str.length());
    printf(")\r\n");
#endif
    return gotit;
}

bool SnCommSBD::WaitForAtResponses(const char* const desired,
                                   const uint32_t timeout,
                                   const char* until) {
    std::string str;
    return WaitForAtResponses(desired, timeout, str, until);
}

bool SnCommSBD::WaitForAtResponses(const char* const desired,
                                   const uint32_t timeout,
                                   std::string& str,
                                   const char* until) {
    // split 'desired' between each ';' and check for
    // any of them during each iteration
    
    // first tokenize
    std::string dstr(desired);
    std::vector<std::string> des;
    size_t pos=0, end=0, lc=0;
    do {
        end = dstr.find(";",pos);
        lc = (end==std::string::npos) ? end : end-pos;
        des.push_back(dstr.substr(pos,lc));
        pos = (end==std::string::npos) ? end : end+1;
    } while (end!=std::string::npos);
/*
#ifdef DEBUG
    printf("desired=%s. waiting for:\r\n",desired);
    std::vector<std::string>::const_iterator ss, send=des.end();
    for (ss=des.begin(); ss!=send; ss++) {
        printf("%s\r\n",ss->c_str());
    }
#endif
*/
    // now look for them
    bool gotit = false;
    do {
        std::vector<std::string>::const_iterator s, dend=des.end();
        for (s=des.begin(); (s!=dend) && (gotit==false); s++) {
            gotit = AtRespContainsTxt(s->c_str(),timeout,str,until);
        }
    } while ((gotit==false)
           && IsTimedOut(timeout)==false);
#ifdef DEBUG
    printf("WaitForAtResponses: returning %s (",
        gotit?"true":"false");
    dispStrBytes(str.c_str(),str.length());
    printf(")\r\n");
#endif
    return gotit;
}

bool SnCommSBD::Connect(const uint32_t timeout) {
#ifdef DEBUG
    printf("SnCommSBD::Connect\r\n");
#endif
    bool connected=false;
    if (fSBD!=0) {
        fSBD->baud( 19200 );
        fSBD->format( 8, Serial::None, 1 );

#ifdef DEBUG
        printf("changed baud to SBD serial\r\n");
        time_t curt = time(0);
        printf("time=%s\r\n",ctime(&curt));
        Timer resett;
        Timer ciert;
        resett.start();
#endif
        // clean up buffers
        ClearSendReceiveBuffers();
        
        // wait for a bit or else the serial port won't be readable
#ifdef USE_RTOS
        Thread::wait(1000);
#else
        wait(1);
#endif
        
        AtCmd("ATZ0", timeout); // reset
        WaitForAtResponse("OK", timeout);
        /* these must be redundant. if the K0 and D0 settings weren't ok,
           we wouldn't be able to talk to the modem to set them anyway
        AtCmd("ATE0", timeout); // no echo
        WaitForAtResponse("OK", timeout);
        // ensure defaults:
        // quiet mode off, text responses, no DTR, software flow control
        // important: setting &K3 or &D2 will break mbed/sbd communications
        AtCmd("ATQ0 V1 &D0 &K0", timeout);
        WaitForAtResponse("OK", timeout);
        */
        // make sure radio is on
        AtCmd("AT*R1", timeout);
        WaitForAtResponse("OK", timeout);
        // clear the mobile originated buffer
        AtCmd("AT+SBDD0", timeout);
        WaitForAtResponses("0;1", timeout);
#ifdef DEBUG
        ciert.start();
#endif
        // wait for network connected signal
        AtCmd("AT+CIER=1,0,1,0", timeout);
        connected = WaitForAtResponse("+CIEV:1,1",timeout);
#ifdef DEBUG
        ciert.stop();
        resett.stop();
        curt = time(0);
        printf("time=%s\r\n",ctime(&curt));
        printf("CIER response time = %d ms\r\n",ciert.read_ms());
        printf("startup resp time  = %d ms\r\n",resett.read_ms());
#endif
    }
    return connected;
}

bool SnCommSBD::PowerDown(const uint32_t timeout) {
    // careful here -- this turns the radio off and it won't
    // come back on unless power is cycled on the modem
#ifdef DEBUG
    printf("SnCommSBD::PowerDown\r\n");
#endif
    // clean up buffers
    ClearSendReceiveBuffers();
    // send power down signal
    AtCmd("AT*F", timeout);
    // wait for OK in case the SBD needs to send a message
    return WaitForAtResponse("OK",timeout);
}

bool SnCommSBD::TrySetSysTimeUnix(const uint32_t timeout,
                                  uint32_t& prvTime,
                                  uint32_t& setTime) {
#ifdef DEBUG
    printf("Getting system time unix\r\n");
#endif
    
    // initiate a session
    AtCmd("AT+SBDIX", timeout);
    WaitForAtResponse("+SBDIX", timeout);
    
    // try to get the system time
    AtCmd("AT-MSSTM", timeout);
    std::string resp;
    Timer corr;
    WaitForAtResponse("MSSTM", timeout, resp);
    corr.start();
    const uint32_t stime = time(0);
    if ( resp.find("no network service")==std::string::npos ) {
        uint32_t ticks(0);
        if (1==sscanf(resp.c_str(),"MSSTM: %x",&ticks)) {
            if (ticks>0) {
                // each tick is 90ms since the iridium epoch
                const double ct = (ticks * kSecPerIridTick) + kIridEpoch;
                const double cct = ceil(ct);
                corr.stop();
                const float wt = cct - ct - corr.read();
                if (wt>0) {
#ifdef USE_RTOS
                    Thread::wait(wt*1000);
#else
                    wait(wt);
#endif
                }
#ifdef DEBUG
                printf("return time %u. ct=%16.16f, cct=%16.16f, wt=%g, corr=%g\r\n",
                    static_cast<uint32_t>(cct), ct, cct, wt, corr.read());
#endif
                setTime = static_cast<uint32_t>(cct);
                set_time(setTime);
                prvTime = stime + static_cast<uint32_t>(wt+0.5);
#ifdef DEBUG
                const time_t nt = time(0);
                printf("time now %u : %s\r\n", nt, ctime(&nt));
#endif                
                
                return true;
            }
        }
    }
    return false;
}

bool SnCommSBD::CheckSignalStrength(const uint32_t timeout,
                                    float& sigstr) {
    // get the (last known) iridium signal strength
    AtCmd("AT+CSQF", timeout);
    std::string csq;
#ifdef DEBUG
    printf("wait for CSQF: to=%u, ctime=%u\r\n",timeout,time(0));
#endif
    WaitForAtResponse("+CSQF", timeout, csq);
    return (1==sscanf(csq.c_str(),"+CSQF:%g",&sigstr));
}



#endif // ENABLE_SBD