/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: Process function calls from various Device states

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Uttam Bhat
*/

#include "LoRaDeviceStateProc.h"
#include "LoRaMacLayerService.h"

eDevicState DeviceState;

sLoRaMacUplinkStatus LoRaMacUplinkStatus;

sLoRaMacDownlinkStatus LoRaMacDownlinkStatus;

LoRaMacPrimitives_t LoRaPrimitives;

LoRaMacCallback_t LoRaCallbacks;

MibRequestConfirm_t LoRaMibReq;

MlmeReq_t mlmeReq;

uint16_t ChannelMaskTemp[6] = {0};

/*!
 * \brief Function executed on TxNextPacket Timeout event
 */
static void OnTxNextPacketTimerEvent( void )
{
    MibRequestConfirm_t mibReq;
    LoRaMacStatus_t status;

    TimerStop( &TxNextPacketTimer );

    mibReq.Type = MIB_NETWORK_JOINED;
    status = LoRaMacMibGetRequestConfirm( &mibReq );

    if( status == LORAMAC_STATUS_OK )
    {
        if( mibReq.Param.IsNetworkJoined == true )
        {
            DeviceState = DEVICE_STATE_SEND;            
        }
        else
        {
            DeviceState = DEVICE_STATE_JOIN;
        }
        NextTx = true;
    }
}

void DeviceInit( void )
{
    LoRaPrimitives.MacMcpsConfirm = McpsConfirm;
    LoRaPrimitives.MacMcpsIndication = McpsIndication;
    LoRaPrimitives.MacMlmeConfirm = MlmeConfirm;
    LoRaCallbacks.GetBatteryLevel = BoardGetBatteryLevel;
    LoRaMacInitialization( &LoRaPrimitives, &LoRaCallbacks );

    TimerInit( &TxNextPacketTimer, OnTxNextPacketTimerEvent );

    LoRaMibReq.Type = MIB_ADR;
    LoRaMibReq.Param.AdrEnable = LORAWAN_ADR_ON;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_PUBLIC_NETWORK;
    LoRaMibReq.Param.EnablePublicNetwork = LORAWAN_PUBLIC_NETWORK;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_CHANNELS_TX_POWER;
    LoRaMibReq.Param.ChannelsTxPower = LORAWAN_TX_POWER;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );
    
    LoRaMibReq.Type = MIB_CHANNELS_MASK;
    LoRaMibReq.Param.ChannelsMask = ChannelMaskTemp;    
    
#ifdef USE_BAND_915_HYBRID_BLOCK_A

    ChannelMaskTemp[0] = 0x00FF;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0001;

#elif defined USE_BAND_915_HYBRID_BLOCK_B

    ChannelMaskTemp[0] = 0xFF00;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0002;

#elif defined USE_BAND_915_HYBRID_BLOCK_C

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0x00FF;    
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0004;

#elif defined USE_BAND_915_HYBRID_BLOCK_D

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0xFF00;    
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0008;

#elif defined USE_BAND_915_HYBRID_BLOCK_E

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0x00FF;    
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0010;

#elif defined USE_BAND_915_HYBRID_BLOCK_F

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0xFF00;    
    ChannelMaskTemp[3] = 0x0000;
    ChannelMaskTemp[4] = 0x0020;

#elif defined USE_BAND_915_HYBRID_BLOCK_G

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0x00FF;    
    ChannelMaskTemp[4] = 0x0040;

#elif defined USE_BAND_915_HYBRID_BLOCK_H

    ChannelMaskTemp[0] = 0x0000;
    ChannelMaskTemp[1] = 0x0000;
    ChannelMaskTemp[2] = 0x0000;
    ChannelMaskTemp[3] = 0xFF00;    
    ChannelMaskTemp[4] = 0x0080;

#elif defined USE_BAND_915

    ChannelMaskTemp[0] = 0xFFFF;
    ChannelMaskTemp[1] = 0xFFFF;
    ChannelMaskTemp[2] = 0xFFFF;
    ChannelMaskTemp[3] = 0xFFFF;
    ChannelMaskTemp[4] = 0x00FF;
    ChannelMaskTemp[5] = 0x0000;

#endif

    LoRaMacMibSetRequestConfirm( &LoRaMibReq );    

    LoRaMacDownlinkStatus.DownlinkCounter = 0;
}

void DeviceJoinUpdate( void )
{
    LoRaMibReq.Type = MIB_NETWORK_JOINED;
    LoRaMacMibGetRequestConfirm( &LoRaMibReq );            
}

void DeviceJoin( void )
{
#if( OVER_THE_AIR_ACTIVATION != 0 )    

    mlmeReq.Type = MLME_JOIN;

    mlmeReq.Req.Join.DevEui = DevEui;
    mlmeReq.Req.Join.AppEui = AppEui;
    mlmeReq.Req.Join.AppKey = AppKey;

    if( NextTx == true )
    {
        LoRaMacMlmeRequest( &mlmeReq );
    }

#else   
    // Choose a random device address if not already defined in Config.h
    if( DevAddr == 0 )
    {
        // Random seed initialization
        srand1( BoardGetRandomSeed( ) );
        DevAddr = randr( 0, 0x01FFFFFF );
    }

    LoRaMibReq.Type = MIB_NET_ID;
    LoRaMibReq.Param.NetID = LORAWAN_NETWORK_ID;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_DEV_ADDR;
    LoRaMibReq.Param.DevAddr = DevAddr;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_NWK_SKEY;
    LoRaMibReq.Param.NwkSKey = NwkSKey;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_APP_SKEY;
    LoRaMibReq.Param.AppSKey = AppSKey;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

    LoRaMibReq.Type = MIB_NETWORK_JOINED;
    LoRaMibReq.Param.IsNetworkJoined = true;
    LoRaMacMibSetRequestConfirm( &LoRaMibReq );

#endif
}

/*!
 * \brief   Prepares the payload of the frame
 */
void PrepareTxFrame( uint8_t port )
{
    MibRequestConfirm_t mibReq;

    if( BoardGetBatteryLevel( ) < LOW_BAT_THRESHOLD )
    {
        mibReq.Type = MIB_CHANNELS_TX_POWER;
        LoRaMacMibGetRequestConfirm( &mibReq );
        // TX_POWER_30_DBM = 0, TX_POWER_28_DBM = 1, ..., TX_POWER_20_DBM = 5, ..., TX_POWER_10_DBM = 10
        // The if condition is then "less than" to check if the power is greater than 20 dBm
        if( mibReq.Param.ChannelsTxPower < TX_POWER_20_DBM )
        {
            mibReq.Param.ChannelsTxPower = TX_POWER_20_DBM;
            LoRaMacMibSetRequestConfirm( &mibReq );
        }
    }
    
    if( port == 224 )
    {
        RunComplianceTest( );        
    }
    else
    {
        PrepareLoRaFrame( port );
    }
}

/*!
 * \brief   Prepares the payload of the frame
 *
 * \retval  [0: frame could be send, 1: error]
 */
bool SendFrame( void )
{
    McpsReq_t mcpsReq;
    LoRaMacTxInfo_t txInfo;
    
    if( LoRaMacQueryTxPossible( AppDataSize, &txInfo ) != LORAMAC_STATUS_OK )
    {
        // Send empty frame in order to flush MAC commands
        mcpsReq.Type = MCPS_UNCONFIRMED;
        mcpsReq.Req.Unconfirmed.fBuffer = NULL;
        mcpsReq.Req.Unconfirmed.fBufferSize = 0;
        mcpsReq.Req.Unconfirmed.Datarate = LORAWAN_DEFAULT_DATARATE;
        
        LoRaMacUplinkStatus.Acked = false;
        LoRaMacUplinkStatus.Port = 0;
        LoRaMacUplinkStatus.Buffer = NULL;
        LoRaMacUplinkStatus.BufferSize = 0;        
    }
    else
    {
        LoRaMacUplinkStatus.Acked = false;
        LoRaMacUplinkStatus.Port = AppPort;
        LoRaMacUplinkStatus.Buffer = AppData;
        LoRaMacUplinkStatus.BufferSize = AppDataSize;

        if( ( IsTxConfirmed == false ) || ( LoRaMacUplinkStatus.UplinkCounter == 0 ) )
        {
            mcpsReq.Type = MCPS_UNCONFIRMED;
            mcpsReq.Req.Unconfirmed.fPort = AppPort;
            mcpsReq.Req.Unconfirmed.fBuffer = AppData;
            mcpsReq.Req.Unconfirmed.fBufferSize = AppDataSize;
            mcpsReq.Req.Unconfirmed.Datarate = LORAWAN_DEFAULT_DATARATE;
        }
        else
        {
            mcpsReq.Type = MCPS_CONFIRMED;
            mcpsReq.Req.Confirmed.fPort = AppPort;
            mcpsReq.Req.Confirmed.fBuffer = AppData;
            mcpsReq.Req.Confirmed.fBufferSize = AppDataSize;
            mcpsReq.Req.Confirmed.NbTrials = 8;
            mcpsReq.Req.Confirmed.Datarate = LORAWAN_DEFAULT_DATARATE;
        }
    }

    LoRaMacUplinkStatus.Type = mcpsReq.Type;

    if( LoRaMacMcpsRequest( &mcpsReq ) == LORAMAC_STATUS_OK )
    {
        return false;
    }
    return true;
}