/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: Define events during Join, Tx & Rx
            Prepare TX packet by appending with appropriate application data

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Uttam Bhat
*/

#include "LoRaEventProc.h"

/*!
 * Defines the application data transmission duty cycle
 */
uint32_t TxDutyCycleTime = APP_TX_DUTYCYCLE;

bool AppLed = 0;

/*!
 * \brief   Prepares the payload of the frame based on application port
 */
void PrepareLoRaFrame( uint8_t port )
{

    switch( port )
    {    
    case 5:
        {   
            uint8_t tmp;
            uint8_t tmpLength;
            uint8_t ptrIndex = 0;
            
            // Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );

            tmp = ( AppLed != 0 ) ? 0x0F : 0x00;
            tmpLength = 1;
            
            LoRaApp.ApplicationAppendData( &tmp, tmpLength ); // Populate lower nibble of 0th Byte with LED state            

            /*!
            *  Read Temperature
            *  Appends 1 Byte to TX buffer
            */
            LoRaApp.ApplicationCall( AppTemp );

            /*!
            *  Read Battery
            *  Appends 1 Byte to TX buffer
            */
            LoRaApp.ApplicationCall( AppBat );

            /*!
            *  Read GPS coordinates
            *  Appends 8 Bytes (3 bytes longitude, 3 bytes latitude, 2 bytes altitude) to TX buffer
            */
            LoRaApp.ApplicationCall( AppGps );            

            /*!
            *  Read Accelerometer
            *  Appends 2 Bytes to TX buffer
            *   Value       Orientation
            *   0x99 0x00   horizontal + faceup
            *   0x66 0x00   horizontal + facedown
            *   0x00 0x11   vertical
            */
            LoRaApp.ApplicationCall( AppAccl ); // Generate Accelerometer data bytes    

            /*!
            *  Generate Ramp data bytes
            *  Appends incremental values of 1 Byte each to TX buffer until Full
            */
            LoRaApp.ApplicationCall( AppRamp );

            break;
        }   

    // Senet M2X ORIENTATION Demo 
    // Set LORAWAN_APP_DATA_SIZE to 2
    case 6: 
        { 
            uint8_t ptrIndex = 1;
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationCall( AppAcclSenet ); // Generate Accelerometer data bytes    
            
            

            break;
        }
        
    /* Senet GPS Demo 
       Data Format (in Hex):
            [01, 02, Lattitude (3 bytes), Longitude (3 Bytes), Elevation (2 bytes), Tx Power (1 byte)]
    */  
    case 7: 
        { 
            uint8_t ptrIndex = 0;
            uint8_t tmp[] = {0x01, 0x02};
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationAppendData( tmp, 2 );  
            
            LoRaApp.ApplicationCall( AppGps ); // Generate Accelerometer data bytes  
            
            uint8_t pow = 30 - 2*(( uint8_t ) LoRaMacUplinkStatus.TxPower);
            LoRaApp.ApplicationAppendData( &pow, 1 );  

            break;
        }

    // Push-Button Demo     
    case 11: 
        { 
            uint8_t ptrIndex = 0;
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationCall( AppPushButton ); // Transmit uplink counter   
            
            break;
        }

    // Transmit on Vertical Orientation Demo     
    case 12: 
        { 
            uint8_t ptrIndex = 0;
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationCall( AppPushButton ); // Transmit uplink counter      
            
            break;
        }

    
    /* Senet myDevices Sensor Demo 
    Data Format (in Hex):
    [Pressure (2 Bytes), Temperature (1 Byte), X-Axis (2 bytes), Y-Axis (2 bytes), Z-Axis (2 bytes), ]
        */  
    case 13: 
        { 
            uint8_t ptrIndex = 0;
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationCall( AppPrsr);

            LoRaApp.ApplicationCall( AppTemp );

            LoRaApp.ApplicationCall( AppBat );
            
            LoRaApp.ApplicationCall( AppAcclSensor ); // Generate Accelerometer data bytes  
                         

            break;
        }
        
    default:
        break;
    }
}


/*!
 * \brief   Sets Interrupt for next payload transmission
 */
void InitNextTxInterrupt( uint8_t port )
{   
    switch( port )
    {  
    /* GPS Application Demo
        Set Timer interrupt for next uplink
    */
    case 5:
        { 
        }

    /* Senet + M2X demo
    Set Timer interrupt for next uplink
    */
    case 6: 
        { 
            
        }
        
    /* Senet GPS Demo 
       Set Timer interrupt for next uplink
    */  
    case 7: 
        { 
            // Schedule next packet transmission
            TxDutyCycleTime = APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND );
            TimerSetValue( &TxNextPacketTimer, TxDutyCycleTime );
            TimerStart( &TxNextPacketTimer );
            break;
        }

    /* Push Button Demo 
       Send Packet Immedietly if PC0 = GND
    */  
    case 11: 
        { 
            volatile bool PushButtonStatus;

            PushButtonStatus = PC0;
            
            if(PushButtonStatus == 0)
            {
                // Send Pkt immedietly if PC = GND
                DeviceState = DEVICE_STATE_SEND;
                NextTx = true;
            }
            else
            {
                // Keep polling
                IsTxIntUpdate = true;
            }
            break;
        }   

    /* Orientation Demo 
       Send Packet Immedietly if Mote is Vertical
    */  
    case 12: 
        {             
            CheckOrientation( );
            
            if(VerticalStatus == true)
            {
                // Send Pkt immedietly if PC = GND
                DeviceState = DEVICE_STATE_SEND;
                NextTx = true;
            }
            else
            {
                // Keep polling
                IsTxIntUpdate = true;
            }
            break;
        }   

    /* Compliance Test 
       Set Timer interrupt for next uplink
    */
    case 224:
        {               
            // Schedule next packet transmission
            TimerSetValue( &TxNextPacketTimer, COMPLIANCE_TX_DUTYCYCLE );
            TimerStart( &TxNextPacketTimer );
            break;
        }

    default:
        {
            // Schedule next packet transmission
            TxDutyCycleTime = APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND );
            TimerSetValue( &TxNextPacketTimer, TxDutyCycleTime );
            TimerStart( &TxNextPacketTimer );
            break;
        }
    }

}

/*!
 * \brief   What to do during JOIN process ? blink/toggle LED etc.
 */
void JoinEvent( void )
{
    // CtrlLED is defined in LoRaBoardAppIf.h
    // param 1: LED color (Red, Yellow or Green)
    // param 2: LED_ON or LED_OFF
    CtrlLED( Red, LED_ON ); 
}


/*!
 * \brief   What to do during TX ? blink/toggle LED etc.
 */
void TxEvent( void )
{
    int blinkTime = 25000;

    // Blink Red LED for 25msec
    BlinkLED( Red, blinkTime );
}

void RxEvent()
{
    // Toggle yellow LED
    ToggleLED( Yellow );  

    // If Rx Data is 0x01 turn on Green LED else if 0x0 Turn Green LED off
    if( LoRaMacDownlinkStatus.BufferSize == 1 )
    {       
        if( LoRaMacDownlinkStatus.Buffer[0] == 0x01 )
        {
            AppLed = 1;
        }
        else
        {
            if( LoRaMacDownlinkStatus.Buffer[0] == 0x00 )
            {
                AppLed = 0;
            }
        }
    }

    if( AppLed != 0 )
    {
        // Turn  USR_LED ON
        CtrlLED( Usr, LED_ON );
    }
    else
    {
        // Turn  USR_LED OFF
        CtrlLED( Usr, LED_OFF );
    }
}

