/**
 * @brief Simple threading using the mbed os demo
 */
#include "mbed.h"
#include "rtos.h"
#include "ST7567.h"

/* LCD and font parameters */
#define LCD_HEIGHT          64
#define LCD_WIDTH           128
#define FONT_HEIGHT         10
#define FONT_WIDTH          5


/* allocate statically stacks for the three threads */
unsigned char rt_stk[1024];
unsigned char hp_stk[1024];
unsigned char lp_stk[1024];

/* creates three tread objects with different priorities */
Thread real_time_thread(osPriorityRealtime, 1024, &rt_stk[0]);
Thread high_prio_thread(osPriorityHigh, 1024, &hp_stk[0]);
Thread low_prio_thread(osPriorityNormal, 1024, &lp_stk[0]);


/* create a semaphore to synchronize the threads */
Semaphore sync_sema;

/* creates a instance of display */
ST7567  disp(D11, D13, D12, D9, D10);




/**
 * @brief real time prio task function 
 */
static void rt_task(void) {
    const char rt_message[] = {"Real time prio Task: \0"};
    
    disp.locate(0, FONT_HEIGHT * 2);        
    
    disp.printf(rt_message);
    
    for(;;) {
        
        /* take the semaphore allowing less priority
         * tasks to run 
         */
         disp.locate(LCD_WIDTH - (sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
         disp.printf("WAIT!");
         sync_sema.wait(0);
        
         /* show the message by 1 second */
         disp.locate(LCD_WIDTH - (sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
         disp.printf("EXEC!");  

         disp.locate(LCD_WIDTH - (sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
         disp.printf("DELAY");
         Thread::wait(50);  
                
    }
    
}

/**
 * @brief high prio task function 
 */
static void hp_task(void){
    const char hp_message[] = {"High priority Task: \0"};
    
    disp.locate(0, FONT_HEIGHT * 3);
    disp.printf(hp_message);
    
    for(;;) {
        /* take the semaphore allowing less priority
         * tasks to run 
         */
         disp.locate(LCD_WIDTH - (sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
         disp.printf("WAIT!");
         sync_sema.wait(0);
        
         /* show the message by 1 second */
         disp.locate(LCD_WIDTH - (sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
         disp.printf("EXEC!");  

         disp.locate(LCD_WIDTH - (sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
         disp.printf("DELAY");
         Thread::wait(50);  

    }
}

/**
 * @brief normal prio task function 
 */
static void np_task(void) {
    
    const char lp_message[] = {"Low priority Task: \0"};
    
    disp.locate(0, FONT_HEIGHT * 4);
    disp.printf(lp_message);

    
    for(;;) {
         /* show the message by 1 second */
         disp.locate(LCD_WIDTH - (sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("EXEC!");  

         disp.locate(LCD_WIDTH - (sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("DELAY");
         Thread::wait(50);  
         
         /* release the semaphore and syncrhonize the tasks */
         disp.locate(LCD_WIDTH - (sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("WAIT!");
         sync_sema.release();

    }
}

/**
 * @brief main loop 
 */
int main(void) {
    const char banner[] = {"Embarcados MBED-OS Thread demo\0"};

    /* configures the display */
    disp.cls();
    disp.set_contrast(0x35);
    disp.locate((LCD_WIDTH - (sizeof(banner) * FONT_WIDTH)) / 2, 0);
    disp.printf(banner);


    /* starts the three tasks */
    real_time_thread.start(rt_task);   
    high_prio_thread.start(hp_task);   
    low_prio_thread.start(np_task);   


    return 0;
}

