/**
 * @brief Simple Semaphore using the mbed os demo
 */
#include "mbed.h"
#include "rtos.h"
#include "ST7567.h"

/* LCD and font parameters */
#define LCD_HEIGHT          64
#define LCD_WIDTH           128
#define FONT_HEIGHT         10
#define FONT_WIDTH          5


/* allocate statically stacks for the three threads */
unsigned char rt_stk[1024];
unsigned char hp_stk[1024];
unsigned char lp_stk[1024];

/* creates three tread objects with different priorities */
Thread real_time_thread(osPriorityRealtime, 1024, &rt_stk[0]);
Thread high_prio_thread(osPriorityHigh, 1024, &hp_stk[0]);
Thread low_prio_thread(osPriorityNormal, 1024, &lp_stk[0]);


/* creates semaphores to explore the capabilites of synchronization */
Semaphore rt_sema;
Semaphore hp_sema;
const char rt_message[] = {"rt_task() sema:\0"};
const char hp_message[] = {"hp_task() sema:\0"};
const char lp_message[] = {"np_task() exec:\0"};



/* creates a instance of display */
ST7567  disp(D11, D13, D12, D9, D10);




/**
 * @brief real time prio task function 
 */
static void rt_task(void) {

    disp.locate(0, FONT_HEIGHT * 2);            
    disp.printf(rt_message);
    rt_sema.wait();               
    
    
    for(;;) {
            
         /* display the semaphore status */
         disp.locate((sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("WAIT!");

         disp.locate((sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
         disp.printf("TOOK!");
         
         /* put a dumming process in order to consume CPU */
         for(int i = 0; i < 0x3FFFFFF; i++) (void)0;
         
         rt_sema.wait();           
    }
    
}

/**
 * @brief high prio task function 
 */
static void hp_task(void){    
    disp.locate(0, FONT_HEIGHT * 3);
    disp.printf(hp_message);
    hp_sema.wait();           

    
    for(;;) {

         /* dsiplay the semaphore status */
         disp.locate((sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("WAIT!");

         disp.locate((sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
         disp.printf("TOOK!");  

         /* put a dumming process in order to consume CPU */
         for(int i = 0; i < 0x3FFFFFF; i++) (void)0;

         hp_sema.wait();       
    }
}

/**
 * @brief normal prio task function 
 */
static void np_task(void) {
        
    disp.locate(0, FONT_HEIGHT * 4);
    disp.printf(lp_message);

    disp.locate((sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
    disp.printf("RUNN!");
    disp.locate((sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
    disp.printf("WAIT!");
    disp.locate((sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
    disp.printf("WAIT!");

    
    for(;;) {
         rt_sema.release();        
         disp.locate((sizeof(rt_message) * FONT_WIDTH), FONT_HEIGHT * 2);
         disp.printf("WAIT!");
         disp.locate((sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("RUNN!");
         Thread::wait(1000);


         hp_sema.release();          
         disp.locate((sizeof(hp_message) * FONT_WIDTH), FONT_HEIGHT * 3);
         disp.printf("WAIT!");
         disp.locate((sizeof(lp_message) * FONT_WIDTH), FONT_HEIGHT * 4);
         disp.printf("RUNN!");
         Thread::wait(1000);
         
    }
}

/**
 * @brief main loop 
 */
int main(void) {
    const char banner[] = {"Embarcados MBED-OS\0"};

    /* configures the display */
    disp.cls();
    disp.set_contrast(0x35);
    disp.locate((LCD_WIDTH - (sizeof(banner) * FONT_WIDTH)) / 2, (LCD_HEIGHT/2) - (FONT_HEIGHT/2));
    disp.printf(banner);
    
    Thread::wait(2500);
    disp.cls();

    /* starts the three tasks */
    real_time_thread.start(rt_task);   
    high_prio_thread.start(hp_task);   
    low_prio_thread.start(np_task);   


    return 0;
}

