/**
 *  @brief NXP FTF LAB2 - Using the UART controls shield with a custom shell
 */


#include "mbed.h"
#include "ST7567.h"


/* LCD screen dimensions */
#define LCD_HEIGHT          64
#define LCD_WIDTH           128

/* LCD font dimensions */
#define FONT_HEIGHT         10
#define FONT_WIDTH          5

/** Instance a on board GLCD object */
 ST7567 glcd(D11, D13, D12, D9, D10);

/** Instance a Led array */
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);


/** Instance a UART class to communicate with pc */
Serial pc_serial(USBTX,USBRX);

/** character buffer */
char serial_buffer[1024] = {0};
char command_buffer[256];
int read_ptr = 0;

/**
 * @brief led command interface 
 */
static void leds_out(int argc, char **argv) {
    if(argc < 2) {
        /* this function must have two arguments */
        pc_serial.printf("## leds: too few arguments, extiging. \n\r");
    } else {
        if(strcmp(argv[0], "on") == 0){
            /* select which led we have to turn on */
            if(strcmp(argv[1], "LED1") == 0) {
                led1 = 0;
            }else if(strcmp(argv[1], "LED2") == 0) {
                led2 = 0;
                
            }else if(strcmp(argv[1], "LED3") == 0) {
                led3 = 0;

            }else {
                pc_serial.printf("leds: invalid arguments, exiting \n\r");
            }
        }else if (strcmp(argv[0], "off") == 0){
            /* select which led we have to turn off */
            
            if(strcmp(argv[1], "LED1") == 0) {
                led1 = 1;

            }else if(strcmp(argv[1], "LED2") == 0) {
                led2 = 1;

            }else if(strcmp(argv[1], "LED3") == 0) {
                led3 = 1;
            }else {
                pc_serial.printf("leds: invalid arguments, exiting \n\r");
            }

        }else {
            pc_serial.printf("leds: invalid arguments, exting \n\r");
        }
    }

}

/**
 * @brief led command interface 
 */
static void display(int argc, char **argv) {
    char disp_buffer[64] = {0};
    int position_x = 0;
    int position_y = 0;    
    
    if(argc < 3) {
        pc_serial.printf("display: Too few arguments, exiting \n\r");
    } else {
        sscanf(argv[0], "%3d", &position_x);
        sscanf(argv[1], "%2d", &position_y);

        /* check position range */
        if(position_x > 127 || position_x < 0) {
            pc_serial.printf("## display: Invalid arguments, exiting");    
            return;
        }

        if(position_y > 63 || position_y < 0) {
            pc_serial.printf("## display: Invalid arguments, exiting");    
            return;
        }

        /* rebuild the string */
        for(int i = 0; i < (argc - 2); i++) {
            strcat(&disp_buffer[0], argv[2 + i]);
            strcat(&disp_buffer[0],(const char *)" ");
        }

        
        /* check consistency */
        //pc_serial.printf("## %3d \n\r", position_x);
        //pc_serial.printf("## %2d \n\r", position_y);
        //pc_serial.printf("## %s \n\r", disp_buffer);
        
        /* prints on the lcd */
        glcd.locate(position_x * FONT_WIDTH, position_y * FONT_HEIGHT);
        glcd.cls();
        glcd.printf(disp_buffer);    
    }

}

/**
 * @brief led command interface 
 */
static void get_info(int argc, char **argv) {
    pc_serial.printf("## *****************************************************\n\r");
    pc_serial.printf("## *   CPU Board: LPCXpresso 4337                      *\n\r");
    pc_serial.printf("## *   Shield Board: General Purpose Shield            *\n\r");
    pc_serial.printf("## *   SOC: NXP LPC4337 Dual Core Soc                  *\n\r");
    pc_serial.printf("## *   CPU1: ARM Cortex M4 (ARMv7M)                    *\n\r");
    pc_serial.printf("## *   CPU1: ARM Cortex M0 (ARMv6M)                    *\n\r");
    pc_serial.printf("## *   CPU Clocks: 120000000 Hz                        *\n\r");
    pc_serial.printf("## *****************************************************\n\r");    

}

/**
 * @brief led command interface 
 */
static void print_usage(void) {
    pc_serial.printf("## use with the syntax below:         \n\r");
    pc_serial.printf("## command <arg1> <arg2> ... <arg16>  \n\r");
    pc_serial.printf("## Available commands:                \n\r");
    pc_serial.printf("## leds    <on|off> <LED1|LED2|LED3> : control the LPC4337 leds \n\r");
    pc_serial.printf("## display  <x> <y> <msg>            : prints [msg] on the on board lcd with optional clear \n\r");        
    pc_serial.printf("## get_info : gets LPC4337 Processor informantion \n\r");
}

/**
 * @brief shell parser  
 */
void shell_parser (char *cmd, int size) {
    int command;
    int cmd_ptr = 0;
    int arg_ptr = 0;
    int cmd_size = 0;
    
    int argc = 0;
    char *argv[16];    
    
    /* copy to the root command */
    memset(&command_buffer, 0, sizeof(command_buffer)); 
    
    /* find the root command terminator (space) */
    while(cmd_ptr < size) {
        if(cmd[cmd_ptr] == ' ') break;
        cmd_ptr++;
    }
    cmd_size = size - cmd_ptr;
    
    
    /* extract command arguments */
    strncpy(&command_buffer[0], &cmd[cmd_ptr + 1], (size - cmd_ptr));    
    
    /* terminates the root command */
    cmd[cmd_ptr] = 0;
    arg_ptr = 0;
    
    //pc_serial.printf("## command: %s \n\r", cmd);
    //pc_serial.printf("## arguments: %s \n\r", command_buffer);

  
    /* extract the further arguments */
    while(arg_ptr < (cmd_size)) {
 
        argc++;
        *(argv + (argc- 1)) = &command_buffer[arg_ptr];
 
        /* find terminator */
        while(command_buffer[arg_ptr] != ' ') {
            arg_ptr++;
        }
        
        /* adds to argument list */
        command_buffer[arg_ptr] = 0;
        arg_ptr++;       
       // pc_serial.printf("## argument no: %d : %s \n\r", argc, argv[argc-1]);
    }



    /* finds and execute the command table */
    if(strcmp("leds", cmd) == 0) { 
        leds_out(argc, argv);
    } else if(strcmp("display", cmd) == 0) {
        display(argc, argv);
    } else if(strcmp("get_info", cmd) == 0) {
        get_info(argc,argv);
    } else {
        print_usage();
    }    

}


/**
 * @brief main application loop
 */
int main(void) 
{   
    const char msg[] = {"Welcome to NXP FTF !\0"};

    /* setup the serial as 115200 bps */
    pc_serial.baud(115200);

    /* setup our on-board glcd */
    glcd.set_contrast(0x35);
    glcd.cls();
    
    /* Center the LCD cursor based on message size*/
    glcd.locate(LCD_WIDTH - (sizeof(msg) * FONT_WIDTH), 
                    (LCD_HEIGHT - FONT_HEIGHT) / 2);

    
    /* prints a welcome message */
    glcd.printf(msg);

    /* initialize the leds */
    led1 = led2 = led3 = 1;
    
    pc_serial.printf("******************************************************************\n\r");    
    pc_serial.printf("***         Welcome to NXP FTF Simple Shell application      ****\n\r");
    pc_serial.printf("*** Type some commands or just Enter key to see the available ****\n\r");
    pc_serial.printf("******************************************************************\n\r");            
    pc_serial.printf(">>");
    
    for(;;) {        
        /* check if we have character available */
        if(pc_serial.readable()) {
            bool new_cmd = false;
            
            /* get the incoming character */
            char c = pc_serial.getc();
                       
            if( (c == '\n') || (c == '\r')) {
                /* handle enter key */
                new_cmd = true;
                pc_serial.printf("\n\r");
                
            }else if( (c == 0x7F) || (c == 0x08)){
                /* handle backspace and del keys */
                pc_serial.printf("\033[1D");
                pc_serial.putc(' ');
                pc_serial.printf("\033[1D");
                
                read_ptr--;
                if(read_ptr < -1) read_ptr = 1023;
                serial_buffer[read_ptr] = ' ';

                
            } else {
                /* loopback the pressed key */
                pc_serial.putc(c);
                
                /* store the incoming character on command circular buffer */
                serial_buffer[read_ptr] = c;
                read_ptr = (read_ptr + 1) % 1024;
            }
            
            
            
            if(new_cmd != false) {
                /* command arrived, has other characters? */
                if(read_ptr != 0) {
                    shell_parser(&serial_buffer[0], read_ptr);
                } else {
                    print_usage();
                }
                /* reset the buffer command */
                memset(&serial_buffer, 0, sizeof(serial_buffer));
                read_ptr = 0;
                pc_serial.printf(">>");
            } 

        }        
    }
}
