/**
 *  @brief NXP FTF LAB1 - measure analog input, control LEDS an prints on GLCD
 */


#include "mbed.h"
#include "ST7567.h"
#include "AnalogIn.h"
#include "PwmOut.h"
#include "PCAL9555.h"
#include "SoftPWM.h"

/* LCD screen dimensions */
#define LCD_HEIGHT          64
#define LCD_WIDTH           128

/* LCD font dimensions */
#define FONT_HEIGHT         10
#define FONT_WIDTH          5

/* Joystick keys mask */
#define JOY_UP_MASK         0x10
#define JOY_DO_MASK         0x01
#define JOY_EN_MASK         0x02


/** Instance a on board GLCD object */
 ST7567 glcd(D11, D13, D12, D9, D10);

/** Instance a Io expander class */
PCAL9555 joy_input(SDA, SCL);
GpioBusIn joy_bus(joy_input,X0_0, X0_1, X0_2, X0_3, X0_4);


/** Instance three PWM Objects */
SoftPWM led_red(LED1);
SoftPWM led_grn(LED2);
SoftPWM led_blu(LED3);

/** Instance a Analog Input */
AnalogIn potentiometer(A3);

/* screen messages */
const char msg[] = {"Welcome to NXP FTF !\0"};
const char led_r_msg[] = {"Led red value:\0"};
const char led_b_msg[] = {"Led blu value:\0"};
const char led_g_msg[] = {"Led grn value:\0"};



/**
 * @brief main application loop
 */
int main(void) 
{   
    int current_cursor = 0, prev_cursor =0;
    bool enter_pressed = false;    

    /* setup our on-board glcd */
    glcd.set_contrast(0x35);
    glcd.cls();
    
    /* Center the LCD cursor based on message size*/
    glcd.locate(LCD_WIDTH - (sizeof(msg) * FONT_WIDTH), 
                    (LCD_HEIGHT - FONT_HEIGHT) / 2);
    
    /* prints a welcome message */
    glcd.printf(msg);
    Thread::wait(2000);
    glcd.cls();
    
    /* prints the initial screen */
    glcd.locate(3*FONT_WIDTH, FONT_HEIGHT * 2);
    glcd.printf(led_r_msg);
    glcd.locate(3*FONT_WIDTH, FONT_HEIGHT * 3);
    glcd.printf(led_g_msg);
    glcd.locate(3*FONT_WIDTH, FONT_HEIGHT * 4);
    glcd.printf(led_b_msg);


    /* put all pwms to a known value at 1KHz period  */

    led_red.period_ms(1);
    led_red.pulsewidth_us(1000);
    led_grn.period_ms(1);
    led_grn.pulsewidth_us(1000);
    led_blu.period_ms(1);
    led_blu.pulsewidth_us(1000);

   
    /* run the main loop at 200ms period */
    for(;;Thread::wait(200)) {
        
        /* read the joystick status */                
        int keys = joy_bus.read();
        if((keys & JOY_DO_MASK) == 0) {
            /* navigate downward */
            current_cursor = (current_cursor + 1) % 3;
        } else if ((keys & JOY_UP_MASK) == 0) {
            /* navigate upward */
            current_cursor--;
            if(current_cursor < 0) current_cursor = 2;
        } else if ((keys & JOY_EN_MASK) == 0) {
        
            /* catch the enter pressed */
            enter_pressed = true;
        } 
        
        if(current_cursor != prev_cursor) {
            /* joystick action ocurred */

            /* erase previous selection */            
            glcd.locate(0, FONT_HEIGHT * (2+prev_cursor));
            glcd.printf("  ");
            
            /* update for next action */
            prev_cursor = current_cursor;
            enter_pressed = false;                                    
        } 
        
        
        /* move the arrow between options */              
        glcd.locate(0, FONT_HEIGHT * (2+current_cursor));
        glcd.printf("->");
        
        if(enter_pressed != false) {
            /* enter pressed, allow editing the field */
            glcd.locate((sizeof(led_r_msg) + 3) * FONT_WIDTH, 
                            FONT_HEIGHT * (2+current_cursor));
            glcd.printf("%5d", potentiometer.read_u16()); 
            
            switch(current_cursor) {
            case 0:{
                    /* update the led duty cicle */
                    int duty = (potentiometer.read_u16() * 1000)/65535;
                    led_red.pulsewidth_us(1000 - duty);
                }
            break;
            
            case 1:{
                    /* update the led duty cicle */
                    int duty = (potentiometer.read_u16() * 1000)/65535;
                    led_blu.pulsewidth_us(1000 - duty);
                }
            break;
            
            case 2:{
                    /* update the led duty cicle */
                    int duty = (potentiometer.read_u16() * 1000)/65535;
                    led_grn.pulsewidth_us(1000 - duty);
                }
            break;
            }  
        }
               
    }  
}
