#include "mbed.h"
#include "EthernetNetIf.h" // ethernet 
#include "NTPClient.h" // Network Time Protocol client

// rudimentary driving of a 20T202DA2JA vacuum fluorescent display, 20 chars x 2 lines
// display has 5 pins
// 1 - green - GND
// 2 - yellow - Vcc
// 3 - orange - SIO - serial data - MOSI - p5
// 4 - red - /STB - active low device select - p10
// 5 - brown - SCK - serial clock - sclk - p7

// Also, I have fitted a battery on my breadboard to the RTC, to keep the time across power downs.

// VFD
SPI spi(p5, p6, p7); // mosi, miso, sclk (miso not used by display)
DigitalOut cs(p10); // chip select (active low)

// alternatively, look on the ethernet interface to find an NTP server and set the time from that.
EthernetNetIf eth; // instantiate an ethernet connection
NTPClient ntp; // instantiate an NTP client

// summertime
int summertime = true; // if this is true, then it's summer!

void DATA(unsigned char displaying) {
    // Select the device by setting chip select low
    cs = 0;
    spi.write(0xfa);
    spi.write(displaying);
    // Deselect the device
    cs = 1;
}

void COM(unsigned char telling) {
    // Select the device by setting chip select low
    cs = 0;
    spi.write(0xf8);
    spi.write(telling);
    // Deselect the device
    cs = 1;
}

int i = 0; // initalise the character steppers

void INITIALISE_DISPLAY() {
    COM(0x01); //clear all display and set DD-RAM address 0 in address counter
    COM(0x02); //move cursor to the original position
    COM(0x06); //set the cursor direction increment and cursor shift enabled
    COM(0x38); //set 8bit operation,2 line display and 100% brightness level
    COM(0x80); //set cursor to the first position of 1st line
    COM(0x0c); //set display on,cursor on,blinking off
}

void printVFDLine1(char statusString[21]) {
    COM(0x80); // first line
    for (i=0; i<20; i++) {
        DATA(statusString[i]); // step through the top line
    }
}

void printVFDLine2(char statusString[21]) {
    COM(0xc0); // second line
    for (i=0; i<20; i++) {
        DATA(statusString[i]); // step through the first part of the bottom line
    }
}

char timeStringTop[21]; // used with the NTP code in the first line display code
char timeStringBottom[21]; // used with the NTP code in the second line display code
char statusString[41]; // place to put startup status text, etc

int main() {
    // Setup the spi for 8 bit data, high steady state clock,
    // second edge capture, with a 1MHz clock rate
    spi.format(8,3); // set SPI mode
    spi.frequency(1000000);

    // Ethernet and NTP stuff begins (from http://mbed.org/cookbook/NTP-Client )
    INITIALISE_DISPLAY();
    char statusString1[21] = "Start               ";
    printVFDLine1(statusString1);
    char statusString2[21] = "Setting up...       ";
    printVFDLine2(statusString2);

    EthernetErr ethErr = eth.setup();
    if (ethErr) {
        printf("Error %d in setup.\n\r", ethErr);

        char statusString3[21] = "Error %d in setup.\n\r";
        printVFDLine1(statusString3);
        return -1;
    }

    Host server(IpAddr(), 123, "0.uk.pool.ntp.org");
    ntp.setTime(server);

    char statusString4[21] = "0.uk.pool.ntp.org   ";
    printVFDLine1(statusString4);
    char statusString5[21] = "Setup OK            ";
    printVFDLine2(statusString5);

    // end of Ethernet and NTP stuff

    INITIALISE_DISPLAY(); // call the function that initialises the VFD

    while (1) {

        time_t rtc_seconds = time(NULL); // somehow gets the current amount of seconds, evah!

        if (summertime) { // if the summertime variable is true, we add 3600 seconds to rtc_seconds
            rtc_seconds = rtc_seconds + 3600;
        }

        // build the two lines of the display separately
        strftime(timeStringTop, 21, "%a %d %b %T ", localtime(&rtc_seconds)); // put the hours:minutes:seconds, then full day, then am/pm
        printVFDLine1(timeStringTop);

        //  char thingStringBottom[21] = "This is a blank line";
        //  printVFDLine2(thingStringBottom);

        wait(1);

    }

}