/*
 * Enviromental Chamber for PCB testing
 * 
 * The Department of Electronics and Digital Systems
 * School of Electrical Engineering, University of Belgrade
 *
 * May 2022.
 *
 */

#include "mbed.h"
#include "Adafruit_GFX.h"
#include "Adafruit_GFX_Config.h"
#include "Adafruit_SSD1306.h"
#include "DHT.h"
#include "DS1820.h"
#include <ctype.h>


// A pin used for temperature sensor
#define DATA_PIN        A0
#define MAX_PROBES      16

// I2C address, 60d or 0x3c:
#define I2C_REAL_ADD                                                        0x3c
#define I2C_ADDRESS                                            I2C_REAL_ADD << 1 

// Set OLED width and heigth [pixel]:
#define OLED_WIDTH_PX                                                        128
#define OLED_HEIGHT_PX                                                        64

// Multipliers of POT1 and POT2 for OLED rectangle position:
#define WIDTH_SCALER                                                         128
#define HEIGHT_SCALER  

// Min and max temp:
#define TEMP_MIN 0
#define TEMP_MAX 70

// UART2 defines
#define COMMAND_LENGTH  3                                                        

// Time delay for acquisition
#define TEMP_DELAY  1.0f

// Time delay for debounce [ms]
#define DEBOUNCE_PER 30

/*
 * Globar variables
 */

// Timer for temperature acquisition
Ticker temp_timer;

// Buttons for relay control
InterruptIn button_sw1 (PC_9);
InterruptIn button_sw2 (PC_8);

// Relay selection signals
DigitalOut relay_select1(PA_10); // D2
DigitalOut relay_select2(PB_3); // D3

// DS1820 sensor object
DS1820* probe[MAX_PROBES];

// Temperature variables
float temp;
int end_temp;

// DS1820 status and acquisition confirmation flag
char data_acquired = 0;

// UART2 flags and variables
char command_received = 0;
char command_buffer[COMMAND_LENGTH] = {0};
char incorrect_command = 0;
char intro_printed = 0;

// Initialize UART for CLI
Serial uart2(PA_2, PA_3);

// Initialize I2C for OLED display
I2C i2c(PB_14,PB_13);

// Initialize OLED display:
Adafruit_SSD1306_I2c myOled(i2c,PB_5,I2C_ADDRESS,OLED_HEIGHT_PX,OLED_WIDTH_PX);

// Declarations for interrupt routines
void ISR_button_sw1 (void);
void ISR_button_sw2 (void);
void ISR_temp(void);
void ISR_Rx(void);

// Declaration of display refresh function
void refresh_display (void);

// Declaration of message parsing function
void parse_message (void);

int main()
{   
    char cmd = 0;
    int ref_temp = 0;
    
    // Initialize OLED:
    myOled.begin();
    
    // Set ISRs for buttons
    button_sw1.fall(&ISR_button_sw1);
    button_sw2.fall(&ISR_button_sw2);
    
    // Initialize probes
    int num_devices = 0;
    while(DS1820::unassignedProbe(DATA_PIN)) {
        probe[num_devices] = new DS1820(DATA_PIN);
        num_devices++;
        if (num_devices == MAX_PROBES)
            break;
    }
    
    printf("Found %d device(s)\r\n\n", num_devices);

    printf("Dobrodosli u Pajinu i Cveletovu komoru!\n\r");
    printf("Spremite se jer polecemo!!1!\n\r");

    while(true)
    {
        if (!intro_printed)
        {
            printf("Molimo, unesite 's' za start: \n\r");
            scanf("%c", &cmd);
            intro_printed = 1;
        }
        if (cmd == 's')
        {
            printf("Molimo, unesite zeljenu temperaturu: \n\r");
            scanf("%d", &ref_temp);
            if (ref_temp > TEMP_MIN && ref_temp < TEMP_MAX)
            {
                temp_timer.attach(&ISR_temp, TEMP_DELAY);      
            }
            cmd = 0;
        }
        
        if (data_acquired)
        {
            refresh_display();
            data_acquired = 0;
        }
    }    
}

// Display refresh function definition
void refresh_display (void)
{
    myOled.clearDisplay();
  //  myOled.printf("It is %3.1foC\r\n", probe.temperature());
    myOled.display();
   // printf("It is %3.1foC\r\n", probe.temperature());
}

// Definition of message parsing function
void parse_message (void)
{
    if (command_buffer[0] == 's')
    {
        if (isdigit(command_buffer[1]) && isdigit(command_buffer[2]))
        {
            end_temp = (command_buffer[1] - 48) * 10 + (command_buffer[2] - 48);
        }
        else
        {
            incorrect_command = 1;    
        }
    }
    else if (command_buffer[0] == 'e')
    {
        
    }
    else
    {
        incorrect_command = 1;
    }    
}

// ISR for temperature acquisition
void ISR_temp (void)
{
    printf("t\n");
    probe[0]->convertTemperature(true, DS1820::all_devices);
    printf("It is %3.1foC\r\n", probe[0]->temperature());
    data_acquired = 1;
}

// ISR for Relay 1 control
void ISR_button_sw1 (void)
{
    wait_ms(DEBOUNCE_PER);
    
    if (!button_sw1)
    {
        relay_select1 = !relay_select1; 
    }
}

// ISR for Relay 2 control
void ISR_button_sw2 (void)
{    
    wait_ms(DEBOUNCE_PER);
    
    if (!button_sw2)
    {
        relay_select2 = !relay_select2; 
    }
}
