//--------------------------------------------------------------
//  IIR フィルタを利用する直交信号発生器のクラス
//
//  2020/07/30, Copyright (c) 2020 MIKAMI, Naoki
//--------------------------------------------------------------

#include "mbed.h"

#ifndef QUAD_PHASE_OSC_IIR_HPP
#define QUAD_PHASE_OSC_IIR_HPP

namespace Mikami
{
    class QuadOscIir
    {
    public:
        // コンストラクタ
        //      f0  周波数，単位：Hz
        //      ts  標本化間隔，単位：μs
        //      amp 振幅
        QuadOscIir(float f0, float ts, float amp = 1)
            : TS_(ts*1.0E-6f), A0_(amp) { Set(f0); }

        // sin/cos 発生
        void Generate(float &sinX, float &cosX)
        {
            float vn = a1_*vn1_ - vn2_;
            sinX = b1S_*vn1_;
            cosX = A0_*vn + b1C_*vn1_;

            vn2_ = vn1_;    // v[n-2] ← v[n-1]
            vn1_ = vn;      // v[n-1] ← v[n]
        }

        // 出力周波数の設定，周波数を変更し再スタートする場合も使える
        //      f0：周波数，単位：Hz
        void Set(float f0)
        {
            static const float PI2 = 6.283185f;
            a1_ = 2.0f*cosf(PI2*f0*TS_);
            b1S_ = A0_*sinf(PI2*f0*TS_);
            b1C_ = -A0_*cosf(PI2*f0*TS_);
            vn1_ = 1.0f;
            vn2_ = 0.0f;
        }

    private:
        const float TS_;        // 標本化間隔
        const float A0_;        // 振幅
        float a1_, b1S_, b1C_;
        float vn1_, vn2_;

        // コピー･コンストラクタ，代入演算子の禁止のため
        QuadOscIir(const QuadOscIir&);
        QuadOscIir& operator=(const QuadOscIir&);
    };
}
#endif  // QUAD_PHASE_OSC_IIR_HPP