/**
 * Interface between Pawn interpreter and mbed platform.
 *
 * Copyright 2011 Pulse-Robotics, Inc.
 * Author: Tyler Wilson
 */

#include <assert.h>

#include "_amxmbed.h"
#include "amxmbed.h"
#include "amx.h"
#include "mbed.h"

Serial* port = 0;

void mbed_set_serial(Serial* serial)
{
    port = serial;
}

int amx_putstr(const char *s)
{
    return port?port->printf(s):0;
}

int amx_putchar(int c)
{
    return port?port->putc(c):0;
}

int amx_fflush(void)
{
    return 0;
}

int amx_getch(void)
{
    return port?port->getc():0;
}

char *amx_gets(char* s, int i)
{
    return port?port->gets(s,i):0;
}

int amx_termctl(int,int)
{
    return 0;
}

void amx_clrscr(void)
{
}

void amx_clreol(void)
{
}

int amx_gotoxy(int x,int y)
{
    return 0;
}

void amx_wherexy(int *x,int *y)
{
}

unsigned int amx_setattr(int foregr,int backgr,int highlight)
{
    return 0;
}

void amx_console(int columns, int lines, int flags)
{
}

void amx_viewsize(int *width,int *height)
{
}

int amx_kbhit(void)
{
    return port?port->readable():0;
}
  
static cell AMX_NATIVE_CALL n_digitalOpen(AMX *amx, const cell *params)
{
    (void)amx;
    DigitalOut* self = new DigitalOut((PinName)params[1]);
//    port->printf("digitalOpen(0x%x) returns 0x%x\n\r", (PinName)params[1], self);
    return (cell)self;
}

static cell AMX_NATIVE_CALL n_digitalRead(AMX *amx, const cell *params)
{
//    port->printf("digitalRead\n\r");
    (void)amx;
    DigitalOut* obj = (DigitalOut*)params[1];
    if (obj)
    {
        return obj->read() != 0;
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_digitalWrite(AMX *amx, const cell *params)
{
    DigitalOut* obj = (DigitalOut*)params[1];
//    port->printf("digitalWrite(0x%x, %d)\n\r", obj, params[2]);
    if (obj)
    {
        obj->write(params[2]);
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_digitalClose(AMX *amx, const cell *params)
{
//    port->printf("digitalClose\n\r");
    //(void)amx;
    DigitalOut* obj = (DigitalOut*)params[1];
    if (obj)
    {
        delete obj;
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_analogInOpen(AMX * amx, const cell *params)
{
    (void)amx;
    AnalogIn* self = new AnalogIn((PinName)params[1]);
    
    return (cell)self;
}

static cell AMX_NATIVE_CALL n_analogInClose(AMX *amx, const cell params[])
{
    (void)amx;
    AnalogIn* obj = (AnalogIn*)params[1];
    
    if(obj)
    {
        return(obj->read());
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_analogInRead(AMX *amx, const cell params[])
{
    (void)amx;
    AnalogIn* obj = (AnalogIn*)params[1];
    
    if(obj)
    {
        float voltage = obj->read();
    
        port->printf("%f read from analog in pin \n\r",voltage);
        return(amx_ftoc(voltage));
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_analogOutOpen(AMX *amx, const cell params[])
{
    (void)amx;
    AnalogOut* self = new AnalogOut((PinName)params[1]);
    
    return (cell)self;
}

static cell AMX_NATIVE_CALL n_analogOutRead(AMX *amx, const cell params[])
{
    (void)amx;
    AnalogOut* obj = (AnalogOut*)params[1];
    
    if(obj)
    {
        float voltage = obj->read();
        port->printf("reading voltage of %f in analog out pin\n\r",voltage);
    
        return(amx_ftoc(voltage));
    }
    
    return 0;
}

static cell AMX_NATIVE_CALL n_analogOutWrite(AMX *amx, const cell params[])
{
    (void)amx;
    AnalogOut* obj = (AnalogOut*)params[1];
    
    if(obj)
    {
        float voltage = amx_ctof(params[2]);
        port->printf("writing voltage of %f to analog out pin\n\r",voltage);
        obj->write(voltage);
    }
    
    return(0);
}

static cell AMX_NATIVE_CALL n_analogOutClose(AMX* amx, const cell params[])
{
    (void)amx;
    AnalogOut* obj = (AnalogOut*)params[1];
    
    if(obj)
    {
        delete obj;
    }
    return 0;
}

#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368)
static cell AMX_NATIVE_CALL n_wait(AMX *amx, const cell *params)
{
    float amount = amx_ctof(params[1]);
    
    wait(amount);
    
    return 0;
}
#endif

static cell AMX_NATIVE_CALL n_wait_ms(AMX *amx, const cell *params)
{
    int amount = (int)params[1];
//    port->printf("waiting %d ms\n\r", amount);
    wait_ms(amount);
    
    return 0;
}

static cell AMX_NATIVE_CALL n_wait_us(AMX *amx, const cell *params)
{
    int amount = (int)params[1];
    wait_us(amount);
    return 0;
}

static cell AMX_NATIVE_CALL n_kbhit(AMX *amx, const cell *params)
{
    return amx_kbhit() != 0;
}


const AMX_NATIVE_INFO mbed_Natives[] = {
  { "digitalOpen",   n_digitalOpen },
  { "digitalRead",   n_digitalRead },
  { "digitalWrite",  n_digitalWrite },
  { "digitalClose",  n_digitalClose },
  { "analogInOpen",  n_analogInOpen },
  { "analogInClose", n_analogInClose },
  { "analogInRead",  n_analogInRead },
  { "analogOutOpen", n_analogOutOpen },
  { "analogOutRead", n_analogOutRead },
  { "analogOutWrite", n_analogOutWrite },
  { "analogOutClose", n_analogOutClose },
  
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368)
  { "wait",   n_wait },   // uses a float, which we do not support on LPC11U24 version
#endif
  { "wait_ms",   n_wait_ms },
  { "wait_us",   n_wait_us },

  { "kbhit",   n_kbhit },
  { NULL, NULL }        /* terminator */
};

int AMXEXPORT AMXAPI amx_mbedInit(AMX *amx)
{
  return amx_Register(amx, mbed_Natives, -1);
}

int AMXEXPORT AMXAPI amx_mbedCleanup(AMX *amx)
{
  (void)amx;
  return AMX_ERR_NONE;
}