/*
 * @file nmea_parser.h
 * @author Tyler Weaver
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * NMEA GPS Serial Output parser.
 * Routine taken from NMEA Software Standard (NMEA 0183)
 * http://www.winsystems.com/software/nmea.pdf
 *
 * Only handles GGA and RMC Messages
 */

#include "mbed.h"

#ifndef MBED_NMEA_PARSER_H
#define MBED_NMEA_PARSER_H

#define NO_LOCK     1
#define NOT_PARSED  2
#define GGA         3
#define GLL         4
#define RMC         5
#define VTG         6

#define PI (3.141592653589793)

/**  A NmeaParser interface for parsing NMEA gps serial output */
class NmeaParser
{
public:

    /**
    * Default Constructor
    * Initalizes variables
    */
    NmeaParser();

    /** Parse the incoming GPS data, returning whether there is a lock
     *
     * Routine from NMEA Software Standard (NMEA 0183)
     *
     * @param line the nmea string to parse, uses tokenizer vs sscanf 
     * @return 1 if there was a lock when the sample was taken (and therefore .longitude and .latitude are valid), else 0
     */
    int parse(char *);
    
    /**
    * @returns quality of signal (0 = No GPS, 1 = GPS, 2 = DGPS)
    */
    int quality();
    /**
    * From RMC message
    * @returns date in ddmmyy format
    */
    int date();
    /**
    * @returns time in utc format hhmmss.ss format
    */
    float utc_time();
    /**
    * @returns Longitude in NMEA format dddmm.mm
    */
    float longitude();
    /**
    * @returns Latitude in NMEA format dddmm.mm
    */
    float latitude();
    /**
    * @returns Altitude Mean Sea Level (MSL) in Meters
    */
    float msl_altitude();
    /**
    * @returns track (heading) made good in degrees true.
    */
    float track();
    /**
    * @returns speed over ground in knots
    */
    float speed();
    /**
    * @returns number of satellites in use
    */
    int satellite_count();

    /**
    * @returns longitude in decimal form (calculated)
    */
    float calc_dec_longitude();
    /**
    * @returns longitude in decimal form (calculated)
    */
    float calc_dec_latitude();
    /**
    * @returns altitude MSL in feet (calculated)
    */
    float calc_altitude_ft();
    /**
    * Initial bearing is the angle to fly at for the shortest flight between two points on a sphere.
    * Calculations from: http://www.movable-type.co.uk/scripts/latlong.html
    *
    * Uses current position and calculates to the inputed point.
    *
    * @param latitude of target point
    * @param longitude of target point
    * @returns initial bearing for flying to given point
    */
    float calc_initial_bearing(float, float);
    /**
    * Uses the shortest distance across a sphere (haversine formula) to calculate distance
    *
    * Uses current position and calculates to the inputed point.
    *
    * @param latitude of target point
    * @param longitude of target point
    * @returns distance in miles to given point
    */
    double calc_dist_to_mi(float, float);
    /**
    * Uses the calc_dist_to_mi then converts to kilometers.
    *
    * @param latitude of target point
    * @param longitude of target point
    * @returns distance in kilometers to given point
    */
    double calc_dist_to_km(float, float);
private:
    float nmea_to_dec(float, char);
    char *my_token(char *,char);

    char stat_string_[128]; // used in my_token
    char *current_;
    
    char *field_[50]; // used by parse nmea

    // calculated values
    volatile float dec_longitude_;
    volatile float dec_latitude_;

    // GGA - Global Positioning System Fixed Data
    volatile float utc_time_;
    volatile float latitude_;
    volatile char lat_reference_;
    volatile float longitude_;
    volatile char long_reference_;
    volatile int quality_;
    volatile int satellite_count_;
    volatile float hdop_;
    volatile float msl_altitude_;
    volatile char msl_altitude_unit_;

    // RMC - Recommended Minimmum Specific GNS Data
    volatile char rmc_status_;
    volatile float speed_;
    volatile float track_;
    volatile int date_;
};

#endif
