/*
 * @file buffered_serial.h
 * @author Tyler Weaver
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Buffered serial UART1 - Setup to work with UART1 (p13,p14)
 *
 * Uses RTOS to block current thread.
 */

#ifndef BUFFERED_SERIAL_H
#define BUFFERED_SERIAL_H

#define BUFFER_SIZE     255
#define LINE_SIZE       80
#define NEXT(x)         ((x+1)&BUFFER_SIZE)
#define IS_TX_FULL      (((tx_in_ + 1) & BUFFER_SIZE) == tx_out_)
#define IS_RX_FULL      (((rx_in_ + 1) & BUFFER_SIZE) == rx_out_)
#define IS_RX_EMPTY     (rx_in_ == rx_out_)

#include "mbed.h"
#include "rtos.h"
/** Buffered serial UART1 - Setup to work with UART1 (p13,p14)
 *
 * Uses RTOS to block current thread.
 */
class BufferedSerial : public Serial
{
public:
    /** Default Constructor
    * Initialize UART1 - Serial(p13,p14)
    * Initialize Semaphores
    * Attach Serial Interrupts
    */
    BufferedSerial();

    /** Put cstring in buffer/output
    *
    * @param cstring to put in buffer for printing (max length = 80 characters)
    */
    void put_line(char*);
    
    /** Gets a cstring from the buffer/input 
    *
    * @param buffer cstring to put line from buffer in (ends at '\n' or 80 characters)
    */
    void get_line(char*);

private:
    void Tx_interrupt();
    void Rx_interrupt();

// for disabling the irq
    IRQn device_irqn;
    
// Circular buffers for serial TX and RX data - used by interrupt routines
// might need to increase buffer size for high baud rates
    char tx_buffer_[BUFFER_SIZE];
    char rx_buffer_[BUFFER_SIZE];
// Circular buffer pointers
// volatile makes read-modify-write atomic
    volatile int tx_in_;
    volatile int tx_out_;
    volatile int rx_in_;
    volatile int rx_out_;
// Line buffers for sprintf and sscanf
    char tx_line_[LINE_SIZE];
    char rx_line_[LINE_SIZE];

    //DigitalOut led1; // debug
    //DigitalOut led2;
    
    Semaphore rx_sem_;
    Semaphore tx_sem_;
};

#endif