/*
* Copyright (c) 2012 Tyler Weaver, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without restriction,
* including without limitation the rights to use, copy, modify, merge, publish, distribute,

* sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mbed.h"

/** TMP36GZ temperature sensor class
*
* @author Tyler Weaver
*
* Example:
* @code
* #include "mbed.h"
* #include "TMP36GZ.h"
*
* TMP36GZ temp_sensor(p20);
* Serial pc(USBTX, USBRX); // tx, rx
*
* int main() {
*     pc.baud(9600);
*     while(1) {
*         pc.printf("Temp: %6.3f deg/C - %6.3f deg/F\n", temp_sensor.sample(), temp_sensor.sample_f());
*         wait(1.0); // wait 1 second
*     }
* }
* @endcode
*
* @section DESCRIPTION
*
* C++ file for TMP36GZ temperature sensor library
*
*  /----\
* |1 2 3 |
* --------
*
* 1 - Vs, connect 3.3V (Vout)
* 2 - Vout - connect to input pin
* 3 - GND - connect to ground
*
* 750mV = 25 deg/C
*
* 10mV / deg/C
*/

class TMP36GZ
{
public:
    /** Constructor for TMP36GZ sensor
    *
    * Analog input pin from p15-p20
    *
    * @param pin the analog input pin (connect to pin 2 on sensor)
    */
    TMP36GZ(PinName pin);

    /** Sample the sensor in deg C
    *
    * @returns float value in deg C
    */
    float sample();

    /** Sample the sensor in deg F
    *
    * @returns float value in deg F
    */
    float sample_f();

private:
    AnalogIn *input_pin_;
};