/**
 * @author Tyler Weaver
 *
 * @section LICENSE
 *
 * Copyright (c) 2012 Tyler Weaver, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * This is a simplified version of the Generation 1 HARP project.
 */

#include "mbed.h"
#include "Servo.h"

#define CENTER  100
#define LEFT    101
#define RIGHT   102

// servos
Servo left_s(p21);
Servo right_s(p22);
Servo release(p23);

// pc serial
Serial pc(USBTX, USBRX);

// xbee serial connection
Serial xbee(p9,p10);

// status leds
BusOut status_led(LED4, LED3, LED2, LED1);

void init(); // initalize all components
void parachute(int);

int main()
{
    init();

    char xbee_input_character;
    int control_state = CENTER;
    bool is_released = false;

    while(1) {
        if(xbee.readable()) {
            xbee_input_character = xbee.getc();

            // switch input
            switch(xbee_input_character) {
                case 'L': // turn left
                    status_led = 0x3;
                    control_state = LEFT;
                    break;
                case 'C': // center
                    status_led = 0x6;
                    control_state = CENTER;
                    break;
                case 'R': // turn right
                    status_led = 0xC;
                    control_state = RIGHT;
                    break;
                case 'X': // release
                    status_led = 0xF;
                    release = 1;
                    control_state = CENTER; // just in case something stupid happened
                    is_released = true;
                    break;
            }
        }
        if(is_released) // dont move the parachute controlls unless released
            parachute(control_state); // run parachute controls
    }
}

/**************************************/
void init()
{
    status_led = 0;
    // pc serial baud
    pc.baud(9600);
    status_led = 0x1;

    // xbee serial baud
    xbee.baud(9600);
    status_led = 0x2;

    // calibrate servos
    pc.puts("Calibrating servos...\t");
    left_s.calibrate_max(0.0014);
    left_s.calibrate_min(-0.0008);
    right_s.calibrate_max(0.0016);
    right_s.calibrate_min(-0.0008);
    release.calibrate(0.0004);
    pc.puts("OK!\n");
    status_led = 0x3;

    // test servos
    pc.puts("Testing servos...\t");
    for(float i = 0.0; i <= 1.0; i+=0.1) {
        left_s = i;
        right_s = i;
        release = i;
        wait(0.5);
    }
    right_s = release = 0;
    left_s = 0;
    pc.puts("OK!\n");
    status_led = 0x4;
}

/**************************************/
void parachute(int state)
{
    switch(state) {
        case CENTER:
            right_s = 0;
            left_s = 0;
            break;
        case RIGHT:
            right_s = 1;
            left_s = 0;
            break;
        case LEFT:
            right_s = 0;
            left_s = 1;
            break;
    }
}