/* Copyright (c) 2012 Tyler Weaver, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 


#include "mbed.h"
#include "rtos.h"
#include "watchdog.h"

// Setup the watchdog timer
Watchdog wdt;

// xbee serial connection
Serial xbee(p13,p14);

// status leds
BusOut status_led(LED4, LED3, LED2, LED1);

typedef struct {
    char msg;   // the direction to turn in
} message_p;

MemoryPool<message_p, 16> mpool_p;
Queue<message_p, 16> queue_p;

void xbee_thread(void const *argument) {
    while (true) {
        if (xbee.readable()) {
            message_p *message = mpool_p.alloc();
            message->msg = xbee.getc();

            queue_p.put(message);
        }
        Thread::wait(100);
    }
}

int main (void) {
    status_led = 0x9;
    // setup watchdog
    wdt.kick(2.0); // 2 second watchdog
    // setup xbee serial
    xbee.baud(9600);

    Thread thread1(xbee_thread);

    while (true) {

        osEvent evt_p = queue_p.get(1000); // wait for 1 second

        if (evt_p.status == osEventMessage) {
            message_p *message = (message_p*)evt_p.value.p;
            printf("\nMessage from xbee: %c\n\r", message->msg);
            switch (message->msg) {
                case 'L': // turn left
                    status_led = 0x3;
                    break;
                case 'C': // center
                    status_led = 0x6;
                    break;
                case 'R': // turn right
                    status_led = 0xC;
                    break;
                case 'F': // flare
                    status_led = 0xF;
                    break;
            }

            mpool_p.free(message);
        }
        wdt.kick();
    }
}
