/**
 9-dof unit testing

 The purpose of this program is to demonstrate and calibrate the three
 sensors on teh 9-doft board.

 The first version of this software will test the ITG3200 gyro to find
 the temperature drift line.  Data will be logged to help determine the
 thermal drift line.
 See: http://mbed.org/users/gltest26/code/ITG3200/wiki/Thermal-Drift

 The second versoin of this will test the HMC5883L 3 axis magnometer

 The thrid version tests the ADXL345 library.
*/

#include "mbed.h"
#include "ITG3200.h"
#include "HMC5883L.h"
#include "ADXL345.h"
//#include "adxl345unit.h"

I2C i2c_bus(p28, p27);
ADXL345 accel(i2c_bus);
HMC5883L magno(i2c_bus);
ITG3200 gyro(i2c_bus);
Timer t;
Serial pc(USBTX, USBRX);
//ADXL345UNIT accel_test(i2c_bus, t);

void sample_time_test();
void itg3200_curve_test();

int main()
{
    itg3200_curve_test();
}

void sample_time_test()
{
    int16_t accel_sample[3];
    int16_t magno_sample[3];
    int16_t gyro_sample[3];
    double temperature;
    double heading_rad, heading_deg;
    wait(0.5); // enough time for everything to prepare a sample
    t.start();
    double accel_time, magno_time, gyro_time, temp_time, head_rad_time, head_deg_time;
    pc.baud(9600);
    
    pc.printf("accel,magno,gyro,temp,head_rad,head_deg\r\n");
    for(int i = 0; i < 100; i++)
    {
        t.reset();
        accel.getXYZ(accel_sample);
        accel_time = t.read();
        magno.getXYZ(magno_sample);
        magno_time = t.read();
        gyro.getXYZ(gyro_sample, ITG3200::Calibration);
        gyro_time = t.read();
        temperature = gyro.getTemperature();
        temp_time = t.read();
        heading_rad = magno.getHeadingXY();
        head_rad_time = t.read();
        heading_deg = magno.getHeadingXYDeg();
        head_deg_time = t.read();
        
        pc.printf("%f,%f,%f,%f,%f,%f\r\n",accel_time,magno_time,gyro_time,temp_time,head_rad_time,head_deg_time);
        
        wait(0.2 - t.read());
    }
}

void ADXL345_test()
{
    int16_t sample[3];
    float elapsed_time;
    float time_stamp[100];
    int16_t samples[100][3];

    Timer t;
    Serial pc(USBTX, USBRX);
    pc.baud(9600);

    accel.setDataRate(ADXL345_100HZ); // 100Hz data rate
    accel.setDataFormatControl(ADXL345_16G | ADXL345_FULL_RES);
    accel.setPowerControl(0x08); // start measurement

    wait(0.001);

    t.start();
    accel.getXYZ(sample);
    elapsed_time = t.read();

    pc.printf("Sample took %f seconds.\r\n", elapsed_time);
    pc.printf("x: %d, y: %d, z: %d\r\n", sample[0],sample[1],sample[2]);

    for(int i = 0; i < 100; i++) {
        t.reset();
        accel.getXYZ(samples[i]);
        time_stamp[i] = t.read();
    }
    for(int i = 0; i < 100; i++) {
        pc.printf("Sample took %f seconds.\r\n", time_stamp[i]);
        pc.printf("x: %d, y: %d, z: %d\r\n", samples[i][0],samples[i][1],samples[i][2]);
    }
}

void hmc5883l_test()
{
    Serial pc(USBTX, USBRX);

    pc.baud(9600);

    HMC5883L compass(i2c_bus);

    int16_t data[3];

    while(1) {
        compass.getXYZ(data);
        pc.printf("x: %4d, y: %4d, z: %4d\r\n", data[0], data[1], data[2]);
        wait(0.067);
    }
}

void itg3200_curve_test()
{
    DigitalOut myled(LED1);
    LocalFileSystem local("local");               // Create the local filesystem under the name "local"
    ITG3200 gyro(p28, p27); // sda, scl - gyro
    
    /*
    const float offset[3] = {99.5, -45.0, -29.7}; // taken from itg3200.xls curve fit test
    const float slope[3] = {-1.05, 0.95, 0.47};

    gyro.setCalibrationCurve(offset, slope);
    gyro.setLpBandwidth(LPFBW_42HZ); 
    */

    Serial pc(USBTX, USBRX);

    pc.baud(9600);

    myled = 0;
    FILE *fp = fopen("/local/itg3200.csv", "w");  // Open "itg3200.csv" for writing
    fputs("Temp, X, Y, Z\r\n", fp);               // place the header at the top

    float temperature = 0.0;
    int16_t gyro_readings[3];

    for(int i = 0; i < 120; i++) { // 120 seconds - 600 samples
        myled = 1;
        gyro.calibrate(1.0);
        myled = 0;
        //gyro.getXYZ(gyro_readings, ITG3200::Calibration);
        gyro.getOffset(gyro_readings);
        temperature = gyro.getTemperature();
        pc.printf("%3d,%f,%d,%d,%d\r\n",i,temperature,gyro_readings[0],gyro_readings[1],gyro_readings[2]);
        fprintf(fp, "%f,%d,%d,%d\r\n",temperature,gyro_readings[0],gyro_readings[1],gyro_readings[2]);
    }
    fclose(fp);
    myled = 0;
}
