/*
 * @file adxl345unit.h
 * @author Tyler Weaver
 *
 * @section LICENSE
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Unit test for the ADXL345 library.
 *
 * Reference: 
 */

#ifndef ADXL345UNIT_H
#define ADXL345UNIT_H

#include "mbed.h"
#include "ADXL345.h"

class ADXL345UNIT {
    public:
        /**
        * Constructor
        * runs init()
        *
        * @param i2c buss to use for adxl345
        */
        ADXL345UNIT(I2C &i2c, Timer &t);
        
        /**
        * Initalize the device
        */
        void init();
        
        /**
        * Perform built in self test and print results to BIST.csv
        * Raw data is printed to BIST_RAW.csv
        *
        * Working
        *
        * @param store_raw true to store raw data
        * @returns test result: true if passed, false if failed
        */
        bool builtInSelfTest(bool);
        
        /**
        * Performs the offset callibration test and prints to OFF_CAL.csv
        *
        * Working
        *
        * Implemented in ADXL345 class
        *
        * @param store_raw true to store raw data
        */
        void offsetCalibration(bool);
        
    private:
        ADXL345 adxl345_;
        Serial pc_;
        DigitalOut open_file_;
        Timer t_;
        
        /**
        * Averages an array of n length
        *
        * @param the array
        * @param length
        */
        int16_t arr_avg(int16_t*,int16_t);
        
        /**
        * Sample 100 times and average
        *
        * @param period of sample rate
        * @param array to hold raw data, should be int16_t[100][3] (sample,axis)
        * @param array to hold averages, should be 3 in length
        * @param pointer to timer object (should already be started)
        */
        void sample100avg(float, int16_t[][3], int16_t*, Timer*);
};

#endif